#include "scope-call-log.h"

#include <glib.h>

#include <telepathy-logger/log-manager.h>
#include <telepathy-logger/text-event.h>
#include <telepathy-logger/call-event.h>

#include <unity.h>

#include "config.h"

#define SCOPE_CALL_LOG_DEFAULT_AVATAR   UNITY_LENS_DIR "/contacts/contacts-lens-default.png"
#define SCOPE_CALL_LOG_DBUS_PATH        "/com/canonical/unity/scope/contacts/calllog"
#define SCOPE_CALL_LOG_CATEGORY_INDEX   1
#define SCOPE_CALL_LOG_MAX_EVENTS       50

#define SCOPE_CALL_LOG_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), SCOPE_CALL_LOG_TYPE, ScopeCallLogPrivate))

G_DEFINE_TYPE (ScopeCallLog, scope_call_log, G_TYPE_OBJECT)

static void _scope_call_log_setup (ScopeCallLog *self);
static void _scope_call_log_on_account_manager_prepared (TpAccountManager *mng, GAsyncResult *res, ScopeCallLog *self);

typedef struct _ScopeCallLogPrivate        ScopeCallLogPrivate;
struct _ScopeCallLogPrivate
{
    TpAccountManager *account_manager;
    TplLogManager *log_manager;
    UnityScope *scope;
    GList *accounts;
    gchar *search_string;

    GCancellable *cancellable;
    UnityLensSearch *search;
    guint entity_threads;
    guint filter_threads;
};

typedef struct _ScopeCallLogSearchData  ScopeCallLogSearchData;
struct _ScopeCallLogSearchData
{
    TpAccount *account;
    ScopeCallLog *self;
};

static void
scope_call_log_dispose (GObject *gobject)
{
    ScopeCallLogPrivate *priv = SCOPE_CALL_LOG_GET_PRIVATE (gobject);

    // TODO: cancel search before destroy class
    g_free (priv->search_string);
    g_list_free (priv->accounts);
    g_object_unref (priv->account_manager);
    g_object_unref (priv->log_manager);

    G_OBJECT_CLASS (scope_call_log_parent_class)->dispose (gobject);
}

static void
scope_call_log_class_init (ScopeCallLogClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

    gobject_class->dispose = scope_call_log_dispose;

    g_type_class_add_private (klass, sizeof (ScopeCallLogPrivate));
}

static void
scope_call_log_init (ScopeCallLog *self)
{
}

ScopeCallLog*
scope_call_log_new (void)
{
    return (ScopeCallLog*) g_object_new (SCOPE_CALL_LOG_TYPE, NULL);
}

void
scope_call_log_run (ScopeCallLog *self)
{
    ScopeCallLogPrivate *priv = SCOPE_CALL_LOG_GET_PRIVATE (self);
    if (!priv->account_manager) {
        priv->account_manager = tp_account_manager_dup ();
    }

    tp_proxy_prepare_async (priv->account_manager,
                            NULL,
                            (GAsyncReadyCallback) _scope_call_log_on_account_manager_prepared,
                            self);
}



static void
_scope_call_log_on_account_manager_prepared (TpAccountManager *mng, GAsyncResult *res, ScopeCallLog *self)
{
    ScopeCallLogPrivate *priv = SCOPE_CALL_LOG_GET_PRIVATE (self);
    GError *error = NULL;

    tp_proxy_prepare_finish (mng, res, &error);
    if (error) {
        g_debug ("Account manager error: %s", error->message);
        g_error_free (error);
    } else {
        priv->accounts = tp_account_manager_get_valid_accounts (mng);
        if (priv->accounts) {
            _scope_call_log_setup (self);
        } else {
            g_warning ("No valid accounts");
        }
    }
}

static gchar*
_scope_call_log_get_formatted_timestamp (guint64 timestamp)
{
    GDateTime *date = g_date_time_new_from_unix_local (timestamp);
    if (date) {
        gchar *format = g_date_time_format (date, "%I:%M %p\n%B %e, %Y");
        g_date_time_unref (date);
        return format;
    } else {
        return g_strdup ("unknown");
    }
}

static void
_scope_call_log_finish_search (ScopeCallLog *self)
{
    ScopeCallLogPrivate *priv = SCOPE_CALL_LOG_GET_PRIVATE (self);

    // Wait for search threads
    if (priv->entity_threads > 0) {
        return;
    }

    if (priv->filter_threads > 0) {
        return;
    }

    g_debug ("Finish search");
    if (priv->cancellable) {
        g_debug ("Clear search vars");
        g_free (priv->search_string);
        priv->search_string = NULL;
        priv->filter_threads = 0;
        priv->entity_threads = 0;
        unity_lens_search_finished (priv->search);
        priv->search = NULL;
        priv->cancellable = NULL;
    }
}

static void
_scope_call_log_on_filter_done (TplLogManager *source, GAsyncResult *res, ScopeCallLog *self)
{
    GError *error = NULL;
    GList *events = NULL;
    ScopeCallLogPrivate *priv = SCOPE_CALL_LOG_GET_PRIVATE (self);

    priv->filter_threads--;
    if (!tpl_log_manager_get_filtered_events_finish (source, res, &events, &error)) {
        g_debug ("Fail to get events: %s", error->message);
        g_error_free (error);
        _scope_call_log_finish_search (self);
        return;
    }

    g_debug ("FOUND %d Events", g_list_length (events));

    DeeSerializableModel *result_model = unity_scope_get_results_model (priv->scope);
    GList *walk = events;

    for(; walk && !g_cancellable_is_cancelled(priv->cancellable); walk = walk->next) {
        TplEvent *event = (TplEvent*) walk->data;
        if (TPL_IS_CALL_EVENT (event)) {
            gint64 timestamp = tpl_event_get_timestamp (event);
            TplEntity *sender = tpl_event_get_sender (event);
            TplEntity *receiver = tpl_event_get_receiver (event);

            TplEntity *buddy = receiver;

            if (tpl_entity_get_entity_type (receiver) == TPL_ENTITY_SELF) {
                buddy = sender;
            }

            const gchar *title = tpl_entity_get_alias (buddy);
            const gchar direction = tpl_entity_get_entity_type (receiver) == TPL_ENTITY_SELF ? 'I' : 'O';
            gchar *comment = NULL;
            if (!title || (strlen (title) == 0)) {
                title = tpl_entity_get_identifier (buddy);
            }
            comment = _scope_call_log_get_formatted_timestamp (timestamp);

            const gchar *id = tpl_entity_get_identifier (buddy);
            gchar *call_uri = g_strdup_printf ("call://%s", id);
            gboolean avatar_exists = FALSE;
            const gchar *avatar_path = tpl_entity_get_avatar_token (buddy);
            if (avatar_path && (strlen(avatar_path) > 0)) {
                GFile *avatar_file = g_file_new_for_path (avatar_path);
                avatar_exists = g_file_query_exists (avatar_file, NULL);
                g_object_unref (avatar_file);
            }
            dee_model_append ((DeeModel*) result_model,
                              call_uri,
                              (avatar_exists ? avatar_path : SCOPE_CALL_LOG_DEFAULT_AVATAR),
                              SCOPE_CALL_LOG_CATEGORY_INDEX,
                              direction == 'I' ? "application/incoming_call" : "application/outgoing_call",
                              title,
                              comment,
                              "");

            g_free (call_uri);
            g_free (comment);

            if (dee_model_get_n_rows((DeeModel*) result_model) >= SCOPE_CALL_LOG_MAX_EVENTS) {
                break;
            }
        }
    }

     _scope_call_log_finish_search (self);
}

static gboolean
_string_contains (const gchar* self, const gchar* needle)
{
    g_return_val_if_fail (self != NULL, FALSE);
    g_return_val_if_fail (needle != NULL, FALSE);

    gchar* self_down = g_ascii_strdown (self, -1);
    gchar* needle_down = g_ascii_strdown (needle, -1);

    gboolean result = (strstr (self_down, needle_down) != NULL);

    g_free (self_down);
    g_free (needle_down);

    return result;
}

static gboolean
_scope_call_log_filter_events (TplEvent *event,
                               ScopeCallLog *self)
{
    ScopeCallLogPrivate *priv = SCOPE_CALL_LOG_GET_PRIVATE (self);

    if (!TPL_IS_CALL_EVENT (event) ||
        g_cancellable_is_cancelled(priv->cancellable)) {
        return FALSE;
    }

    gchar *search_string = priv->search_string;

    if (!search_string || (strlen(search_string) == 0)) {
        g_debug ("Search string is empty");
        return TRUE;
    }

    TplEntity *sender = tpl_event_get_sender (event);
    TplEntity *receiver = tpl_event_get_receiver (event);

    TplEntity *buddy = receiver;

    if (tpl_entity_get_entity_type (receiver) == TPL_ENTITY_SELF) {
        buddy = sender;
    }

    const gchar *title = tpl_entity_get_alias (buddy);
    const gchar *id = tpl_entity_get_identifier (buddy);

    if (_string_contains (title, search_string) ||
        _string_contains (id, search_string)) {
        return TRUE;
    }

    return FALSE;
}


static void
_scope_call_log_get_entities_done (TplLogManager *source, GAsyncResult *res, ScopeCallLogSearchData *data)
{
    ScopeCallLog *self = data->self;
    ScopeCallLogPrivate *priv = SCOPE_CALL_LOG_GET_PRIVATE (data->self);
    TpAccount *account = data->account;
    GList *entities = NULL;
    GError *error = NULL;

    g_free (data);

    priv->entity_threads--;
    if (!tpl_log_manager_get_entities_finish (source, res, &entities, &error)) {
        g_warning ("Fail to get entities related with account: %s", error->message);
        g_error_free (error);
        _scope_call_log_finish_search (self);
        return;
    }

    GList *walk = entities;
    for (; walk && !g_cancellable_is_cancelled(priv->cancellable); walk = walk->next) {
        priv->filter_threads++;
        tpl_log_manager_get_filtered_events_async (priv->log_manager,
                                                   account,
                                                   (TplEntity*) walk->data,
                                                   TPL_EVENT_MASK_CALL,
                                                   SCOPE_CALL_LOG_MAX_EVENTS,
                                                   (TplLogEventFilter) _scope_call_log_filter_events,
                                                   self,
                                                   (GAsyncReadyCallback) _scope_call_log_on_filter_done,
                                                   self);
    }

    if ((g_list_length (entities) == 0) || g_cancellable_is_cancelled (priv->cancellable)) {
        g_debug ("FINISH: _scope_call_log_get_entities_done %d %d", g_list_length (entities),
                 g_cancellable_is_cancelled (priv->cancellable));
        _scope_call_log_finish_search (self);
    }
}

static void
_scope_call_log_on_search_changed (UnityScope *scope,
                                   UnityLensSearch *search,
                                   UnitySearchType type,
                                   GCancellable *cancellable,
                                   ScopeCallLog *self)
{
    ScopeCallLogPrivate *priv = SCOPE_CALL_LOG_GET_PRIVATE (self);

    if (g_list_length (priv->accounts) == 0) {
        g_debug ("No accounts");
        unity_lens_search_finished (search);
        return;
    }

    g_return_if_fail (priv->cancellable == NULL);
    priv->search = search;
    priv->cancellable = cancellable;

    priv->search_string = g_strdup (unity_lens_search_get_search_string (search));
    g_debug ("Call log search: [%s]\n", priv->search_string);

    DeeSerializableModel *result_model = unity_scope_get_results_model (priv->scope);
    dee_model_clear ((DeeModel*) result_model);

    GList *walk = priv->accounts;
    for(; walk && !g_cancellable_is_cancelled (priv->cancellable); walk = walk->next) {
        ScopeCallLogSearchData *data = g_new0 (ScopeCallLogSearchData, 1);
        data->self = self;
        data->account = (TpAccount*) walk->data;
        priv->entity_threads++;
        tpl_log_manager_get_entities_async (priv->log_manager,
                                            data->account,
                                            (GAsyncReadyCallback) _scope_call_log_get_entities_done,
                                            data);
    }

    if (g_cancellable_is_cancelled (priv->cancellable)) {
        g_debug ("FINISH: _scope_call_log_on_search_changed");
        _scope_call_log_finish_search (self);
    }
}

static void
_scope_call_log_on_activate (UnityScope *scope,
                             GParamSpec *spec,
                             ScopeCallLog *self)
{
}

static void
_scope_call_log_setup (ScopeCallLog *self)
{
    ScopeCallLogPrivate *priv = SCOPE_CALL_LOG_GET_PRIVATE (self);

    priv->log_manager = tpl_log_manager_dup_singleton ();
    priv->scope = unity_scope_new (SCOPE_CALL_LOG_DBUS_PATH);
    unity_scope_set_search_in_global (priv->scope, FALSE);
    g_signal_connect (priv->scope, "search-changed", (GCallback)_scope_call_log_on_search_changed, self);
    g_signal_connect (priv->scope, "notify::active", (GCallback)_scope_call_log_on_activate, self);

    GError *error = NULL;
    unity_scope_export (priv->scope, &error);
    if (error) {
        g_debug ("Fail to export scope: %s", error->message);
        g_error_free (error);
        g_clear_object (&priv->scope);
    }
}
