/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: PropertiesImpl.cxx,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: fridrich_strba $ $Date: 2007/04/26 21:34:43 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

/*  Copyright 2005 Sun Microsystems, Inc. */

#include "PropertiesImpl.hxx"
#include <odiapi/props/Logger.hxx>
#include <boost/assert.hpp>
#include <stdlib.h>
#include <stdio.h>
#include <stdexcept>
#include <utility>
#include <memory>
#include <algorithm>
#include <errno.h>

using namespace boost;
using namespace std;
using namespace util;

#ifdef WNT
#define SNPRINTF(buffer, size, format, args) _snprintf(buffer, size, format, args)
#else
#define SNPRINTF(buffer, size, format, args) snprintf(buffer, size, format, args)
#endif

namespace /* private */
{
    using namespace ::odiapi::props;
    using namespace ::writerfilter;

    const string UNIT_CM = string("cm");
    const string UNIT_MM = string("mm");
    const string UNIT_IN = string("in");
    const string UNIT_PT = string("pt");
    const string UNIT_PC = string("pc");
    const string UNIT_PX = string("px");
  
    string intToString(int i)
    {
        char buff[12];
        SNPRINTF(buff, sizeof(buff), "%i", i);
        return string(buff);  
    }

    string floatToString(double f)
    {
        char buff[100];
        SNPRINTF(buff, sizeof(buff), "%.3f", f);
        return string(buff);  
    }

    inline const char* skipSpaces(const char* pch)
    {
        while (*pch && *pch == ' ')
            pch++;
        return pch;
    }

    inline bool isLeadingSign(const char ch)
    {	
        return (ch == '-' || ch == '+');
    }

    /** Dump property pool in a format usable bei Graphviz and the Dot tool 
        (see http://www.graphviz.org)
    */
    string substituteChars(const string& s, const string& subStr, const string& replaceWith)
    {
        string tmpStr = s;
        string::size_type pos = tmpStr.find(subStr);
        while (pos != string::npos)
        {
            tmpStr.replace(pos, replaceWith.size(), replaceWith);
            pos = tmpStr.find(subStr);
        }
        return tmpStr;
    }
  
    string uniqueIdOfNode(Node* node, Node* rootNode)
    {
        return node != rootNode ? intToString(reinterpret_cast<sal_IntPtr>(node)) : string("Root");	
    }

    string displayNameOfNode(Node* node, Node* rootNode)
    {
        return node != rootNode ? QName::tokenizer().getLocalName(node->mProperty->getId()) : string("Root");
    }

    string valueOfNode(Node* node, Node* rootNode)
    {
        string val = displayNameOfNode(node, rootNode);
        ArrayPropertyImpl* pa = dynamic_cast<ArrayPropertyImpl*>(node->mProperty.get());            
        if (pa)
            val = val + string(", ") + intToString(pa->getPos());

        return val + (node->mProperty.get() && !node->mProperty->getStringValue().empty() ? 
                      (string(", ") + substituteChars(node->mProperty->getStringValue(), string(" "), string("_"))) : string());
    }

    string refersToNodeId(Node* node, Node* rootNode)
    {
        CompositePropertyImpl* pc = dynamic_cast<CompositePropertyImpl*>(node->mProperty.get());
        if (!pc)
        {
            ArrayPropertyImpl* pa = dynamic_cast<ArrayPropertyImpl*>(node->mProperty.get());
            if (pa)
                pc = dynamic_cast<CompositePropertyImpl*>(pa->getWrappedProperty().get());
        }
        return pc ? uniqueIdOfNode(pc->refersTo(), rootNode) : string();        
    }

    void logNode(Node* node, Node* rootNode, util::Logger* logger)
    {
        logger->beginNode(uniqueIdOfNode(node, rootNode), valueOfNode(node, rootNode), refersToNodeId(node, rootNode), node->isInUse());
      
        odiapi::props::NodeContainer_t::iterator iter = node->mChildren.begin();
        odiapi::props::NodeContainer_t::iterator iter_end = node->mChildren.end();
        for (; iter != iter_end; ++iter)
            logNode(*iter, rootNode, logger);
      
        logger->endNode(uniqueIdOfNode(node, rootNode));     
    }

    void deleteNode(Node* node)
    {
        delete node;
    }

    /** 
        @return the node that must be deleted next 
    */
    Node* deleteUnusedNodes(Node* node)
    {
        Node* deleteThisNode = NULL;
		
        for (NodeContainer_t::iterator iter = node->mChildren.begin(); 
             iter != node->mChildren.end(); 
             ++iter)
        {
            /* In place modification of the container while 
               iterating allowed because the container type 
               is a set */
            Node* deleteThisNode1 = deleteUnusedNodes(*iter);

            if (deleteThisNode1 != NULL)
            {
                node->mChildren.erase(deleteThisNode);

                delete deleteThisNode1;
            }
        }
	
        if (!node->isInUse() && node->mChildren.empty())
        {	  	  
            deleteThisNode = node;
        }
        return deleteThisNode;
    }


} // namespace private


namespace odiapi { namespace props {
  
#ifdef DEBUG
    /* static */ int Node::mDebugNodeCount = 0;
    /* static */ int PropertyPoolImpl::debugUseCount_ = 0;
#endif

    PropertyImpl::PropertyImpl(QName_t id) :
        mId(id)
    {}

    PropertyImpl::~PropertyImpl()
    {}

    QName_t PropertyImpl::getId() const
    {
        return mId;
    }

    int PropertyImpl::getIntValue() const
    {
        throw logic_error("Unsupported operation");
    }

    Property::Pointer_t PropertyImpl::getChild(int) const
    {
        throw logic_error("Unsupported operation");
    }

    Property::Pointer_t PropertyImpl::findChild(QName_t) const
    {
        throw logic_error("Unsupported operation");
    }

    PropertyBag_Pointer_t PropertyImpl::getPropertyBag() const
    {
        throw logic_error("Unsupported operation");
    }

    bool PropertyImpl::hasPropertyBag() const
    {
        return false;
    }

    bool PropertyImpl::less(const Property& other) const
    {
        return (this->getId() < other.getId());
    }
	
void PropertyImpl::resolve(PropertyPoolHandle_Pointer_t /*realReference*/) const
    {
        throw logic_error("Unsupported operation");
    }


    IntPropertyImpl::IntPropertyImpl(QName_t id, int value) :
        PropertyImpl(id),
        mValue(value)
    {}

    IntPropertyImpl::IntPropertyImpl(QName_t id) :
        PropertyImpl(id)
    {}

    void IntPropertyImpl::setValue(int value)
    {
        mValue = value;
    }

    int IntPropertyImpl::getIntValue() const
    {
        return mValue;
    }

    string IntPropertyImpl::getStringValue() const
    {
        return intToString(mValue);
    }

    bool IntPropertyImpl::less(const Property& other) const
    {
        const IntPropertyImpl* otherInt = dynamic_cast<const IntPropertyImpl*>(&other);	
        bool isThisLessThanOther = false;
    
        if (PropertyImpl::less(other))
            isThisLessThanOther = true;
        else if (other.getId() == this->getId() && this->getIntValue() < otherInt->getIntValue()) 
            isThisLessThanOther = true;

        return isThisLessThanOther;
    }

    double TwipsPropertyImpl::valueInCm(const string& valueAndUnit) const
    {
        const char* pValStart = skipSpaces(valueAndUnit.c_str());		
        const char* pValEnd = pValStart;

        if (!(isdigit(*pValEnd) || isLeadingSign(*pValEnd)))
            throw invalid_argument("Invalid value provided");

        while (*pValEnd && (isdigit(*pValEnd) || isLeadingSign(*pValEnd) || *pValEnd == '.'))
            pValEnd++;
		   
        string value(pValStart, pValEnd);	

        errno = 0;
        double vdbl = strtod(value.c_str(), (char**)NULL);		
        if (vdbl == 0 && errno != 0)
            throw std::invalid_argument("Invalid value provided");

        const char* pUnitStart = skipSpaces(pValEnd);	
        const char* pUnitEnd = pUnitStart;
        while (*pUnitEnd && isalpha(*pUnitEnd)) 
            pUnitEnd++;

        string unit(pUnitStart, pUnitEnd);
	
        double conversion_factor = getConversionFactor(unit);

        if (!(conversion_factor > 0))
            throw std::invalid_argument("Invalid unit provided");

        return (vdbl * conversion_factor);
    }

    double TwipsPropertyImpl::getConversionFactor(const string& unit) const
    {
        double cf = 0.0;

        if (unit == UNIT_CM)
            cf = 1;
        else if (unit == UNIT_MM)
            cf = 0.1;
        else if (unit == UNIT_IN)
            cf = 2.54;
        else if (unit == UNIT_PT)
            cf = 0.0352778;
        else if (unit == UNIT_PC)
            BOOST_ASSERT(false && "Not yet implemented");
        else if (unit == UNIT_PX)
            BOOST_ASSERT(false && "Not yet implemented");

        return cf;
    }

    TwipsPropertyImpl::TwipsPropertyImpl(QName_t id, const std::string& valueAndUnit) :
        IntPropertyImpl(id)
    {
        // convert value and unit to twips - 1cm = 567 Twip
        double valueInTwips = valueInCm(valueAndUnit) * 567;
        setValue(static_cast<int>(valueInTwips));
    }

    TwipsPropertyImpl::TwipsPropertyImpl(QName_t id, int valueInTwips) :
        IntPropertyImpl(id, valueInTwips)
    {}

    const string MEASUREMENT_UNIT_CM = (" cm");
    string TwipsPropertyImpl::getStringValue() const
    {
        // convert current twips value in 'cm' - 1 cm = 567 Twip
        string valueInCm_ = floatToString(double(getIntValue()) / double(567));
        return valueInCm_ + MEASUREMENT_UNIT_CM;
    }

    StringPropertyImpl::StringPropertyImpl(QName_t id, const string& value) :
        PropertyImpl(id),
        mValue(value)
    {}

    string StringPropertyImpl::getStringValue() const
    {
        return mValue;
    }

    bool StringPropertyImpl::less(const Property& other) const
    {
        const StringPropertyImpl* otherString = dynamic_cast<const StringPropertyImpl*>(&other);	
        bool isThisLessThanOther = false;
	
        if (PropertyImpl::less(other))
            isThisLessThanOther = true;
        else if (other.getId() == this->getId() && this->getStringValue() < otherString->getStringValue()) 
            isThisLessThanOther = true;

        return isThisLessThanOther;
    }

    CompositePropertyImpl::CompositePropertyImpl(QName_t id, PropertyPoolHandle_Pointer_t poolHandle) :
        PropertyImpl(id),
        mPoolHandle(poolHandle)
    {}

    string CompositePropertyImpl::getStringValue() const
    {
        return string("");
    }

    bool CompositePropertyImpl::less(const Property& other) const
    {
        const CompositePropertyImpl* otherComposite = dynamic_cast<const CompositePropertyImpl*>(&other);
        bool isThisLessThanOther = false;

        if (PropertyImpl::less(other))
            isThisLessThanOther = true;
        else if (other.getId() == this->getId() && this->mPoolHandle < otherComposite->mPoolHandle)
            isThisLessThanOther = true;

        return isThisLessThanOther;
    }

    Property::Pointer_t CompositePropertyImpl::getChild(int pos) const
    {
        return mPoolHandle->getPropertyBag()->get(pos);
    }

    Property::Pointer_t CompositePropertyImpl::findChild(QName_t id) const
    {
        return mPoolHandle->getPropertyBag()->find(id);
    }

    PropertyBag_Pointer_t CompositePropertyImpl::getPropertyBag() const
    {
        return mPoolHandle->getPropertyBag();
    }

    bool CompositePropertyImpl::hasPropertyBag() const
    {
        return true;
    }

    Node* CompositePropertyImpl::refersTo()
    {
        return dynamic_cast<PropertyPoolHandleImpl*>(mPoolHandle.get())->refersTo();
    }
    
    void CompositePropertyImpl::setPoolHandle(PropertyPoolHandle_Pointer_t poolHandle) const
    {
        mPoolHandle = poolHandle;
    }

    bool CompositePropertyImpl::isValidPoolHandle() const
    {
        return mPoolHandle.get()!=NULL;
    }


    ReferencePropertyImpl::ReferencePropertyImpl(QName_t id, QName_t /*referenceProxy*/) :
        CompositePropertyImpl(id, PropertyPoolHandle_Pointer_t()),
        referenceProxy(referenceProxy)
    {}

    string ReferencePropertyImpl::getStringValue() const
    {
        if (isValidPoolHandle())
            return CompositePropertyImpl::getStringValue();
		else
			throw logic_error("Reference not resolved!!!");
    }


    bool ReferencePropertyImpl::less(const Property& other) const
    {
        const ReferencePropertyImpl* otherComposite = dynamic_cast<const ReferencePropertyImpl*>(&other);
        bool isThisLessThanOther = false;

        if (PropertyImpl::less(other))
            isThisLessThanOther = true;
        else if (other.getId() == this->getId() && this->referenceProxy < otherComposite->referenceProxy) 
            isThisLessThanOther = true;

        return isThisLessThanOther;
    }

    Property::Pointer_t ReferencePropertyImpl::getChild(int pos) const
    {
        if (isValidPoolHandle())
        {
            return CompositePropertyImpl::getChild(pos);
        }
        else
        {
            throw logic_error("Reference not resolved!!!");
        }
    }

    Property::Pointer_t ReferencePropertyImpl::findChild(QName_t id) const
    {
        if (isValidPoolHandle())
        {
            return CompositePropertyImpl::findChild(id);
        }
        else
        {
            throw logic_error("Reference not resolved!!!");
        }
    }

    PropertyBag_Pointer_t ReferencePropertyImpl::getPropertyBag() const
    {
        if (isValidPoolHandle())
        {
            return CompositePropertyImpl::getPropertyBag();
        }
        else
        {
            throw logic_error("Reference not resolved!!!");
        }
    }

    bool ReferencePropertyImpl::hasPropertyBag() const
    {
        if (isValidPoolHandle())
        {
            return CompositePropertyImpl::hasPropertyBag();
        }
        else
        {
            throw logic_error("Reference not resolved!!!");
        }
    }

    Node* ReferencePropertyImpl::refersTo()
    {
        if (isValidPoolHandle())
        {
            return CompositePropertyImpl::refersTo();
        }
        else
        {
            throw logic_error("Reference not resolved!!!");
        }
    }
    

    void ReferencePropertyImpl::resolve(PropertyPoolHandle_Pointer_t realReference) const
    {
        setPoolHandle(realReference);
    }


    ArrayPropertyImpl::ArrayPropertyImpl(size_t pos, Property::Pointer_t property) :
        PropertyImpl(NS_rtf::LN_li),
        mPos(pos),
        mWrappedProperty(property)
    {
    }

    PropertyBag_Pointer_t ArrayPropertyImpl::getPropertyBag() const
    {
        return mWrappedProperty->getPropertyBag();
    }

    bool ArrayPropertyImpl::hasPropertyBag() const
    {
        return mWrappedProperty->hasPropertyBag();
    }

    size_t ArrayPropertyImpl::getPos() const
    {
        return mPos;
    }

    int ArrayPropertyImpl::getIntValue() const
    {
        return mWrappedProperty->getIntValue();
    }

    string ArrayPropertyImpl::getStringValue() const
    {
        return mWrappedProperty->getStringValue();
    }

    bool ArrayPropertyImpl::less(const Property& other) const
    {	
        bool isThisLessThanOther = false;

        if (PropertyImpl::less(other))
        {
            isThisLessThanOther = true;
        }
        else if (other.getId() == this->getId()) 
        {
            const ArrayPropertyImpl* otherArray = dynamic_cast<const ArrayPropertyImpl*>(&other);
            
            if (this->mPos < otherArray->mPos)
                isThisLessThanOther = true;
            else if (this->mPos == otherArray->mPos)
                isThisLessThanOther = this->mWrappedProperty->less(*otherArray->mWrappedProperty.get());
        }
        return isThisLessThanOther;
    }

    Property::Pointer_t ArrayPropertyImpl::getWrappedProperty()
    {
        return mWrappedProperty;
    }

    bool operator<(const Property::Pointer_t& first, const Property::Pointer_t& second)
    {
        return first->less(*second.get());
    }

    bool operator==(const Property::Pointer_t& first, const Property::Pointer_t& second)
    {
        return (!(first < second) && !(second < first));
    }
  
    bool operator!=(const Property::Pointer_t& first, const Property::Pointer_t& second)
    {
        return !(first == second);
    }

    PropertyBagImpl::PropertyBagImpl() :
        mIdentity(UNDEFINED)
    {}

    PropertyBagImpl::~PropertyBagImpl()
    {}

    size_t PropertyBagImpl::size() const
    {
        return mProperties.size();
    }

    void PropertyBagImpl::clear()
    {
        mProperties.clear();
        mIdentity = UNDEFINED;
    }

    PropertyBag_Pointer_t PropertyBagImpl::copy() const
    {
        return PropertyBag_Pointer_t(new PropertyBagImpl(*this));
    }

    void PropertyBagImpl::insert(Property::Pointer_t property)
    {
        BOOST_ASSERT(mIdentity != ARRAY && "Already used as array must not be used as struct");
        mIdentity = STRUCTURE;

        ensurePropertyNotExist(property->getId());
        mProperties.insert(PropertyContainer_t::value_type(property->getId(), property));
    }
 
    Property::Pointer_t PropertyBagImpl::find(QName_t id) const
    {
        const_iterator iter = mProperties.find(id);
        return (iter != mProperties.end()) ? (*iter).second : Property::Pointer_t();
    }

    void PropertyBagImpl::insert(int pos, Property::Pointer_t property)
    {
        BOOST_ASSERT(mIdentity != STRUCTURE && "Already used as structure must not be used as array");
        mIdentity = ARRAY;

        ensurePropertyNotExist(pos);
        // When used as array we have to insert an array property
        mProperties.insert(PropertyContainer_t::value_type(pos, 
                                                           Property::Pointer_t(new ArrayPropertyImpl(pos, property))));
    }

    Property::Pointer_t PropertyBagImpl::get(int pos) const
    {
        BOOST_ASSERT(((mIdentity == ARRAY) || (mIdentity == UNDEFINED)) && "Not used as array method must not be used");

        const_iterator iter = mProperties.find(pos);
        Property::Pointer_t prop;

        if (iter != mProperties.end())
        {
            ArrayPropertyImpl* parr = dynamic_cast<ArrayPropertyImpl*>((*iter).second.get());
            BOOST_ASSERT(parr && "Must be an ArrayPropertyImpl");
            prop = parr->getWrappedProperty();
        }
        return prop; 
    }

    void PropertyBagImpl::sort()
    {
        // nothing to do as we are using a std::map 
        // which is a sorted container already
    }

    Iterator<Property::Pointer_t>::Pointer_t PropertyBagImpl::createIterator()
    {
        return Iterator<Property::Pointer_t>::Pointer_t(new PropertyBagIteratorImpl(*this));
    }

    void PropertyBagImpl::ensurePropertyNotExist(int pos)
    {
        iterator iter = mProperties.find(pos);
        if (iter != mProperties.end())
            mProperties.erase(iter);
    }
    
    PropertyBagIteratorImpl::PropertyBagIteratorImpl(PropertyBagImpl& propertyBag) :
        mPropertyBag(propertyBag),
        mIter(propertyBag.mProperties.begin())
    {}
	

    void PropertyBagIteratorImpl::first() 
    {
        mIter = mPropertyBag.mProperties.begin();
    }

    void PropertyBagIteratorImpl::next()
    {
        ++mIter;
    }

    Property::Pointer_t PropertyBagIteratorImpl::getCurrent() const
    {
		ArrayPropertyImpl* parr = dynamic_cast<ArrayPropertyImpl*>((*mIter).second.get());
		return parr ? parr->getWrappedProperty() : (*mIter).second;
    }

    bool PropertyBagIteratorImpl::isDone() const
    {
        return mIter == mPropertyBag.mProperties.end();
    }


    Node::Node() :
        mReferenceCount(0),
        mParent(0)
    {
        init();
    }

    Node::Node(Property::Pointer_t property) :
        mReferenceCount(0),
        mProperty(property),
        mParent(0)
    {
        init();
    }

    Node::~Node()
    {	
        // Notify all listener that we're about to be disposed
        for_each(mListener.begin(), mListener.end(), mem_fun(&WeakReference::disposing));
        BOOST_ASSERT(!isInUse() && "There are still references on this node");
        for_each(mChildren.begin(), mChildren.end(), deleteNode); 

#ifdef DEBUG
        mDebugNodeCount--;		
#endif
    }
  
    void Node::acquire()
    {
        mReferenceCount++;
    }

    void Node::release()
    {
        mReferenceCount--;
    }

    void Node::registerWeakReference(WeakReference* ref)
    {
        mListener.insert(ref);
    }

    void Node::unregisterWeakReference(WeakReference* ref)
    {
        WeakReferenceContainer_t::iterator iter = mListener.find(ref);
        if (iter != mListener.end())
            mListener.erase(iter);
    }

    void Node::init()
    {
#ifdef DEBUG
        mDebugNodeCount++;
#endif
    }

    bool Node::isInUse()
    {
        return mReferenceCount != 0;
    }

    bool Node::hasChildren() const
    {
        return !mChildren.empty();
    }

    bool lessNodePtr::operator()(const Node* first, const Node* second) const
    {
        return first->mProperty < second->mProperty;
    }

    PropertyPoolHandleImpl::PropertyPoolHandleImpl(Node* node) :
        mNode(node)
    {
        mNode->acquire();
        mNode->registerWeakReference(this);
    }
  
    PropertyPoolHandleImpl::~PropertyPoolHandleImpl()
    {
        if (mNode)
        {
            mNode->release();
            mNode->unregisterWeakReference(this);	
        }
    }

    PropertyBag_Pointer_t PropertyPoolHandleImpl::getPropertyBag() const 
    {
        BOOST_ASSERT(mNode && "Invalid pool handle");
        return mNode->mCachedPropertyBag;
    }

    /** We must no longer refer to the node after this method 
     */
    void PropertyPoolHandleImpl::disposing()
    {
        mNode->release();
        mNode = NULL;
    }

    Node* PropertyPoolHandleImpl::refersTo()
    {
        return mNode;
    }

    PropertyBagsIteratorImpl::PropertyBagsIteratorImpl(const PropertyPoolImpl& pool) :
        mPool(pool)
    {
    }

    void PropertyBagsIteratorImpl::first()
    {	
        mIter = mPool.mUsedNodes.begin();
        goToNextUsed();
    }

    void PropertyBagsIteratorImpl::next()
    {
        ++mIter;
        goToNextUsed();
    }

    void PropertyBagsIteratorImpl::goToNextUsed()
    {
        PropertyPoolImpl::NodeList_t::const_iterator iter_end = mPool.mUsedNodes.end();
	
        // travers to the next leaf node still in use
        for (; mIter != iter_end && !(*mIter)->isInUse(); ++mIter) 
        { 
            /* NOP */
        }	 
    }

    PropertyBag_Pointer_t PropertyBagsIteratorImpl::getCurrent() const 
    { 
        return (*mIter)->mCachedPropertyBag;
    }

    bool PropertyBagsIteratorImpl::isDone() const
    {
        return mIter == mPool.mUsedNodes.end();
    }


    PropertyPoolImpl::PropertyPoolImpl() : 
        mRootNode(new Node)
    {
        mRootNode->acquire();
        mRootNode->mCachedPropertyBag = PropertyBag_Pointer_t(new PropertyBagImpl);

#ifdef DEBUG
        debugUseCount_++;
#endif
    }

    PropertyPoolImpl::~PropertyPoolImpl()
    {	
        mRootNode->release();
        delete mRootNode;

#ifdef DEBUG
        debugUseCount_--;	
#endif
    }

    typedef auto_ptr<Node> NodePtr_t;

    PropertyPoolHandle_Pointer_t PropertyPoolImpl::insert(PropertyBag_Pointer_t propertyBag)
    {
        BOOST_ASSERT(propertyBag.get() && "Invalid property sequence provided");
  
        PropertyBag_Pointer_t orderedProperties = propertyBag->copy();
        orderedProperties->sort();

        Iterator<Property::Pointer_t>::Pointer_t iter = orderedProperties->createIterator();        
        Node* currentNode = mRootNode;

        for (iter->first(); !iter->isDone(); iter->next())
        {
            NodePtr_t newNode(new Node(iter->getCurrent()));
	  
            NodeContainer_t::iterator node_iter = currentNode->mChildren.find(newNode.get());
            if (node_iter == currentNode->mChildren.end()) 
            { // Node not yet in pool
                newNode->mParent = currentNode;
                currentNode->mChildren.insert(newNode.get());	  		
                currentNode = newNode.release();
            }
            else // Node already in pool
            { 
                currentNode = *node_iter;
            }
        }
        // the leafs of the tree cache the provided PropertyBag 
        if (currentNode->mCachedPropertyBag.get() == NULL)
        {
            currentNode->mCachedPropertyBag = orderedProperties; 
            mUsedNodes.push_back(currentNode);
        }

        return PropertyPoolHandle_Pointer_t(new PropertyPoolHandleImpl(currentNode));
    }

    Iterator<PropertyBag_Pointer_t>::Pointer_t PropertyPoolImpl::createIterator() const
    {
        return Iterator<PropertyBag_Pointer_t>::Pointer_t(new PropertyBagsIteratorImpl(*this));
    }

    void PropertyPoolImpl::unregisterNodesFromUsedNodesList()
    {	
        NodeList_t::iterator iter = mUsedNodes.begin();
        while (iter != mUsedNodes.end()) 
        {
            if (!(*iter)->isInUse())
                iter = mUsedNodes.erase(iter);
            else 
                ++iter;
        }
    }

    void PropertyPoolImpl::garbageCollection()
    {
        unregisterNodesFromUsedNodesList();
        deleteUnusedNodes(mRootNode);
    }

    void PropertyPoolImpl::dump(Logger* logger) const
    {
        logger->beginTree();
        logNode(mRootNode, mRootNode, logger);
        logger->endTree();
    }

    size_t PropertyPoolImpl::dbgGetNumberOfNodes() 
    {
#ifdef DEBUG
        return Node::mDebugNodeCount;
#else
        return 0;
#endif
    }
  
    void PropertyPoolImpl::dbgCountUsedNodes(Node* node, size_t* count) 
    {
        if (node->isInUse())
            *count++;

        NodeContainer_t::const_iterator iter = node->mChildren.begin();
        NodeContainer_t::const_iterator iter_end = node->mChildren.end();
        for (; iter != iter_end; ++iter)
            dbgCountUsedNodes(*iter, count);
    }

    size_t PropertyPoolImpl::dbgGetNumberOfUsedNodes() 
    {
        size_t count = 0;
        dbgCountUsedNodes(mRootNode, &count);
        return count;
    }
  
    bool operator<(const PropertyPoolHandle_Pointer_t& first, const PropertyPoolHandle_Pointer_t& second)
    {
        return (dynamic_cast<PropertyPoolHandleImpl*>(first.get())->mNode < 
                dynamic_cast<PropertyPoolHandleImpl*>(second.get())->mNode);
    }

    bool operator==(const PropertyPoolHandle_Pointer_t& first, const PropertyPoolHandle_Pointer_t& second)
    {
        return (!(first < second) && !(second < first));
    }

    bool operator!=(const PropertyPoolHandle_Pointer_t& first, const PropertyPoolHandle_Pointer_t& second)
    {
        return !(first == second);
    }

    Property::Pointer_t createIntegerProperty(QName_t id, int value) 
    {
        return Property::Pointer_t(new IntPropertyImpl(id, value));
    }

    Property::Pointer_t createStringProperty(QName_t id, const string& value) 
    {
        return Property::Pointer_t(new StringPropertyImpl(id, value));
    }
 
    Property::Pointer_t createCompositeProperty(QName_t id, PropertyPoolHandle_Pointer_t poolHandle)
    {
        return Property::Pointer_t(new CompositePropertyImpl(id, poolHandle));
    }

    Property::Pointer_t createTwipsProperty(writerfilter::QName_t id, const std::string& valueAndUnit)
    {
        return Property::Pointer_t(new TwipsPropertyImpl(id, valueAndUnit));
    }

    Property::Pointer_t createTwipsProperty(writerfilter::QName_t id, int valueInTwips)
    {
        return Property::Pointer_t(new TwipsPropertyImpl(id, valueInTwips));
    }

    Property::Pointer_t createReferenceProperty(QName_t id, QName_t referenceProxy)
    {
        return Property::Pointer_t(new ReferencePropertyImpl(id, referenceProxy));
    }

    PropertyBag_Pointer_t createPropertyBag()
    {
        return PropertyBag_Pointer_t(new PropertyBagImpl());
    }

    PropertyPool::Pointer_t createPropertyPool()
    {
        return PropertyPool::Pointer_t(new PropertyPoolImpl());
    }

} } // namespace odiapi { namespace props {
