/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 2; tab-width: 2 -*- */
/*
 * Copyright (C) 2009 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Gordon Allott (gord.allott@canonical.com)
 */
#include "launcher-application.h"
#include "launcher-appman.h"
#include <sys/inotify.h>

/**
 * SECTION:launcher-appman
 * @short_description: Singleton object designed to cache
 * and hold #LauncherApplication objects
 * @include: launcher-appman.h
 *
 * #LauncherAppman is a singleton that will serve as a single point of entry
 * for creating #LauncherApplication objects, all #LauncherApplication objects
 * should be requested from this cache by providing a desktop file to the
 * application you wish to create.
 */

G_DEFINE_TYPE (LauncherAppman, launcher_appman, G_TYPE_OBJECT);
#define LAUNCHER_APPMAN_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE(obj, \
LAUNCHER_TYPE_APPMAN, LauncherAppmanPrivate))

enum
{
  PROP_0,

  PROP_ENABLE_WINDOW_CHECKING,
};

enum
{
  WATCH_FILE_MODIFIED,

  LAST_SIGNAL
};

static guint appman_signals[LAST_SIGNAL] = { 0 };
struct _LauncherAppmanPrivate
{
  GSequence *app_list;
  gint        in_fd;  // Inotify file descriptor
  guint       in_watch_handler; // g_io_watch event handler for inotify
  GIOChannel *in_io_channel; // GIOChannel for the inotify file descriptor
  gboolean    enable_window_checking;
};

static LauncherAppman *launcher_appman = NULL;

static gboolean
on_inotify_update (GIOChannel *source, GIOCondition condition)
{

  char            buffer[32*1024] __attribute__ ((aligned)); //32kb buffer - don't want more than that really
  GError          *error = NULL;
  gsize           len;
  guint           index;
  LauncherAppman  *appman = launcher_appman_get_default ();

  g_return_val_if_fail (LAUNCHER_IS_APPMAN (appman), FALSE);
  g_io_channel_read_chars (source, buffer, 32*1024, &len, &error);
  if (error != NULL)
    {
      g_warning ("INotify error: %s", error->message);
      return FALSE;
    }

  index = 0;
  while (index < len)
    {
      // loop through the buffer and grab as many structures as we can
      struct inotify_event *event;
      event = (struct inotify_event *) &buffer[index];
      // need to emit a signal for each event we get
      g_signal_emit (appman, appman_signals[WATCH_FILE_MODIFIED], 0, event->wd);
      index += sizeof (struct inotify_event) + event->len;
    }
  return TRUE;

}

static void
launcher_appman_finalize (GObject *appman)
{
  LauncherAppmanPrivate *priv;
  g_return_if_fail (LAUNCHER_IS_APPMAN (appman));
  priv = LAUNCHER_APPMAN (appman)->priv;

  /* we assume that the application is shutting down and no longer needs the
   * app cache
   */
  g_sequence_free(priv->app_list);

  G_OBJECT_CLASS (launcher_appman_parent_class)->finalize (appman);
}

static void
launcher_appman_init (LauncherAppman *appman)
{
  LauncherAppmanPrivate *priv;
  GError *error = NULL;
  priv = appman->priv = LAUNCHER_APPMAN_GET_PRIVATE (appman);

  // init inotify
  priv->in_fd = inotify_init ();
  priv->in_io_channel = g_io_channel_unix_new (priv->in_fd);
  g_io_channel_set_encoding (priv->in_io_channel, NULL, NULL);
  g_io_channel_set_flags (priv->in_io_channel, G_IO_FLAG_NONBLOCK, &error);
  if (error != NULL)
    {
      g_warning ("could not set flags on iochannel: %s", error->message);
    }
  priv->in_watch_handler = g_io_add_watch (priv->in_io_channel,
                                           G_IO_IN,
                                           (GIOFunc)on_inotify_update,
                                           NULL);

  /* FIXME - add GDestroyNotify peramater */
  priv->app_list = g_sequence_new(NULL);
}


static void
launcher_appman_set_property (GObject *object,
                              guint prop_id,
                              const GValue *value,
                              GParamSpec *pspec)
{
  LauncherAppman *appman = LAUNCHER_APPMAN(object);
  LauncherAppmanPrivate *priv = appman->priv;
  g_return_if_fail (LAUNCHER_IS_APPMAN (appman));

  switch (prop_id)
    {
    case PROP_ENABLE_WINDOW_CHECKING:
      priv->enable_window_checking = g_value_get_boolean(value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
launcher_appman_get_property (GObject *object,
                                   guint prop_id,
                                   GValue *value,
                                   GParamSpec *pspec)
{
  LauncherAppman *appman = LAUNCHER_APPMAN(object);
  LauncherAppmanPrivate *priv = appman->priv;

  g_return_if_fail (LAUNCHER_IS_APPMAN (appman));

  switch (prop_id)
    {
    case PROP_ENABLE_WINDOW_CHECKING:
      g_value_set_boolean (value, priv->enable_window_checking);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
launcher_appman_class_init (LauncherAppmanClass *klass)
{
  GObjectClass* object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = launcher_appman_finalize;

  object_class->set_property = launcher_appman_set_property;
  object_class->get_property = launcher_appman_get_property;

  g_type_class_add_private (object_class, sizeof (LauncherAppmanPrivate));

  g_object_class_install_property (object_class,
                                   PROP_ENABLE_WINDOW_CHECKING,
                                   g_param_spec_boolean ("enable-window-checking",
                                                         "enable-window-checking",
                                                         "will enable communication with wncksync when creating launcher-applications",
                                                         FALSE,
                                                         G_PARAM_READABLE|G_PARAM_WRITABLE));

  appman_signals[WATCH_FILE_MODIFIED] =
      g_signal_new  ("watch-file-modified",
                    G_OBJECT_CLASS_TYPE (klass),
                    0,
                    0, NULL, NULL,
                    g_cclosure_marshal_VOID__INT,
                    G_TYPE_NONE, 1, G_TYPE_INT
                    );
}

static void
launcher_appman_add_application (LauncherAppman *appman,
                                 LauncherApplication *application)
{
  LauncherAppmanPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_APPMAN (appman));
  g_return_if_fail (LAUNCHER_IS_APPLICATION (application));

  priv = appman->priv;
  g_sequence_append (priv->app_list, application);
}



/*
 * Public methods
 */

/**
 * launcher_appman_get_default
 *
 * Returns the default LauncherAppman singleton
 *
 * Returns: A #LauncherApppman Object
 */
LauncherAppman *
launcher_appman_get_default (void)
{
  if (G_UNLIKELY (!LAUNCHER_IS_APPMAN (launcher_appman)))
    {
      launcher_appman = g_object_new (LAUNCHER_TYPE_APPMAN,
                                       NULL);
      return launcher_appman;
    }

  return g_object_ref (launcher_appman);
}

/**
 * launcher_appman_get_application_for_desktop_file
 * @appman: A #LauncherAppman object
 * @desktop: A String path to the desktop file
 *
 * Provides a method for accessing #LauncherApplication objects by providing
 * a desktop file
 *
 * Returns: A LauncherApplication object or NULL
 */
LauncherApplication *
launcher_appman_get_application_for_desktop_file (LauncherAppman *appman,
                                                  const gchar *desktop)
{
  LauncherAppmanPrivate *    priv;
  LauncherApplication *   app;
  const gchar *           app_desktop_file;
  GSequenceIter *         iter;

  g_return_val_if_fail (LAUNCHER_IS_APPMAN (appman), NULL);

  priv = appman->priv;

  for (iter = g_sequence_get_begin_iter (priv->app_list);
       !g_sequence_iter_is_end (iter);
       iter = g_sequence_iter_next (iter))
    {
      app = g_sequence_get (iter);

      if (!LAUNCHER_IS_APPLICATION (app))
        continue;

      app_desktop_file = launcher_application_get_desktop_file (app);

      if (g_strcmp0 (app_desktop_file, desktop) == 0)
        return app;
    }

  app = launcher_application_new_from_desktop_file (desktop, !priv->enable_window_checking);
  launcher_appman_add_application (appman, app);

  return app;
}

/**
 * launcher_appman_get_application_for_wnck_app
 * @appman: A #LauncherAppman object
 * @name: A WnckApplication
 *
 * Provides a method for accessing #LauncherApplication objects by providing
 * a WnckApplication object
 *
 * Returns: A LauncherApplication object or NULL
 */
LauncherApplication *
launcher_appman_get_application_for_wnck_window (LauncherAppman *appman,
                                                 WnckWindow     *wnck_window)
{
  LauncherAppmanPrivate *    priv;
  LauncherApplication *      app;
  GSequenceIter *            iter;

  g_return_val_if_fail (LAUNCHER_IS_APPMAN (appman), NULL);
  g_return_val_if_fail (WNCK_IS_WINDOW (wnck_window), NULL);
  // we need this method because of complications todo with sometimes not having
  // a desktop file available

  priv = appman->priv;

  for (iter = g_sequence_get_begin_iter (priv->app_list);
       !g_sequence_iter_is_end (iter);
       iter = g_sequence_iter_next (iter))
    {
      app = g_sequence_get (iter);

      if (!LAUNCHER_IS_APPLICATION (app))
        continue;

      if (launcher_application_owns_window (app, wnck_window))
        return app;
    }

  app = launcher_application_new_from_wnck_window (wnck_window);
  launcher_appman_add_application (appman, app);

  return app;
}


/**
 * launcher_appman_get_applications
 * @appman: a #LauncherAppman object
 *
 * returns a #GSequence object that contains all the applications in
 * the current cache
 *
 * Returns: (transfer none): An unowned #GSequence of #LauncherApplication
 * objects
 */
GSequence *
launcher_appman_get_applications (LauncherAppman *appman)
{
  g_return_val_if_fail (LAUNCHER_IS_APPMAN (appman), NULL);
  return appman->priv->app_list;
}


/**
 * launcher_appman_add_file_watch
 * @appman: a #LauncherAPpman object
 * @file_path: a string to the file path
 *
 * Adds a watch on the specified file at file path, will watch for modifications
 * to the file.
 *
 * Returns: an integer watch descriptor or 0 if failed
 */
gint
launcher_appman_add_file_watch (LauncherAppman *appman, const gchar *path)
{
  g_return_val_if_fail (LAUNCHER_IS_APPMAN (appman), 0);
  // set a watch on our desktop file
  return inotify_add_watch(appman->priv->in_fd, path, IN_MODIFY | IN_MOVED_TO | IN_CLOSE_WRITE);
}

/**
 * launcher_appman_rm_file_watch
 * @appman: a #LauncherAPpman object
 * @wd: the integer watch descriptor
 *
 * removes the given @wd the watch
 */
void
launcher_appman_rm_file_watch (LauncherAppman *appman, gint wd)
{
  g_return_if_fail (LAUNCHER_IS_APPMAN (appman));
  if (wd == 0) return;
  // remove the old watch on our desktop file
  inotify_rm_watch (appman->priv->in_fd, wd);
}

void
launcher_appman_set_enable_window_checking (LauncherAppman *appman,
                                            gboolean enable_window_checking)
{
  g_return_if_fail (LAUNCHER_IS_APPMAN (appman));
  appman->priv->enable_window_checking = enable_window_checking;
}

gboolean
launcher_appman_get_enable_window_checking (LauncherAppman *appman)
{
  g_return_val_if_fail (LAUNCHER_IS_APPMAN (appman), FALSE);
  return appman->priv->enable_window_checking;
}
