/*
 * Copyright (C) 2009 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "launcher-category.h"

struct _LauncherCategory
{
  gchar  *name;
  gchar  *comment;
  gchar  *icon_name;
  GSList *children;
};

static LauncherCategory *
launcher_category_copy (const LauncherCategory *category)
{
  if (G_LIKELY (category != NULL))
    {
      LauncherCategory *ret = NULL;

      ret = g_slice_new0 (LauncherCategory);
      /*FIXME: Update for real copy */
      return ret;
    }
  return NULL;
}

void
launcher_category_free (LauncherCategory *category)
{
  if (G_LIKELY (category != NULL))
    {
      g_free (category->name);
      g_free (category->comment);
      g_free (category->icon_name);
      g_slist_free (category->children);
      category->children = NULL;

      g_slice_free (LauncherCategory, category);
    }
}

GType
launcher_category_get_type (void)
{
  static GType _launcher_category_type = 0;

  if (G_UNLIKELY (_launcher_category_type == 0))
    {
      _launcher_category_type =
        g_boxed_type_register_static ("LauncherCategory",
                                      (GBoxedCopyFunc) launcher_category_copy,
                                      (GBoxedFreeFunc) launcher_category_free);

    }

  return _launcher_category_type;
}

/*
 * Public Methods
 */
LauncherCategory *
launcher_category_new (const gchar *name,
                       const gchar *comment,
                       const gchar *icon_name)
{
  LauncherCategory *category;

  category = g_slice_new0 (LauncherCategory);

  if (category != NULL)
    {
      category->name = g_strdup (name);
      category->comment = g_strdup (comment);
      category->icon_name = g_strdup (icon_name);
      category->children = NULL;
    }

  return category;
}

const gchar *
launcher_category_get_name (LauncherCategory *category)
{
  g_return_val_if_fail (category, NULL);
  return category->name;
}

const gchar *
launcher_category_get_comment (LauncherCategory *category)
{
  g_return_val_if_fail (category, NULL);
  return category->comment;
}

const gchar *
launcher_category_get_icon_name (LauncherCategory *category)
{
  g_return_val_if_fail (category, NULL);
  return category->icon_name;
}

void
launcher_category_append_child (LauncherCategory *category,
                                gpointer     child)
{
  g_return_if_fail (category);
  g_return_if_fail (child);

  category->children = g_slist_append (category->children, child);
}

void
launcher_category_sort_children (LauncherCategory *category,
                                 GCompareFunc      sort_func)
{
  g_return_if_fail (category);
  g_return_if_fail (sort_func);

  category->children = g_slist_sort (category->children, sort_func);
}

GSList *
launcher_category_get_children (LauncherCategory *category)
{
  g_return_val_if_fail (category, NULL);

  return category->children;
}

void
launcher_category_empty_children (LauncherCategory *category)
{
  g_return_if_fail (category);

  g_slist_free (category->children);
  category->children = NULL;
}
