/*
 * Copyright 2005-2009 Freescale Semiconductor, Inc. All Rights Reserved.
 *
 * Based on gstavidemux.c by Erik Walthinsen <omega@temple-baptist.com>.
 */

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * Module Name:    mfw_gst_avi_demuxer.c
 *
 * Description:    AVI Demuxer plug-in for Gstreamer.
 *
 * Portability:    This code is written for Linux OS and Gstreamer
 */  
 
/*
 * Changelog: 
 * 
 */



/*=============================================================================
							INCLUDE FILES
=============================================================================*/
#include <gst/gst.h>
#include <memory.h>
#include "avi_parser/AVI_Demux.h"
#include "gst/riff/riff-media.h"
#include "mfw_gst_avidemuxer.h"
#include "mfw_gst_utils.h"

/*=============================================================================
							LOCAL CONSTANTS
=============================================================================*/
#define INVALID_READ_SIZE 5000000

 
#define MFW_AVI_HEADER_SIZE     8
#define WAVE_FORMAT_ALAW        0x0006
#define WAVE_FORMAT_MPEG        0X0050
#define WAVE_FORMAT_MPEG_LAYER3 0X0055

#define FORWARD     1
#define BACKWARD    2
#define TRICK_TIME  (500*GST_MSECOND)

#define DEMUX_WAIT_INTERVAL 10000


/*=============================================================================
				LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
=============================================================================*/

static GstStaticPadTemplate mfw_gst_avidemuxer_sink_template_factory
    = GST_STATIC_PAD_TEMPLATE("sink",
			      GST_PAD_SINK,
			      GST_PAD_ALWAYS,
			      GST_STATIC_CAPS("video/x-msvideo")
    );

/*=============================================================================
										LOCAL MACROS
=============================================================================*/
/* Used for debugging */
#define GST_CAT_DEFAULT mfw_gst_avidemuxer_debug

/* define a tag name */
#define	MFW_GST_TAG_WIDTH		        "width"
#define MFW_GST_TAG_HEIGHT	            "height"
#define MFW_GST_TAG_FRAMERATE           "framerate"
#define MFW_GST_TAG_SAMPLING_FREQUENCY  "sampling frequency"
#define MFW_GST_TAG_PRODUCT             "product"
#define MFW_GST_TAG_SUBJECT             "subject"

#define MAX_INDEX_ENTRY_COUNT   0x200000

#define AVI_ENTRY_INDEX_NR(avi, entry)  ((entry) - (avi)->index_entries)
#define AVI_ENTRY_CALC_TIMESTAMP(demuxer, entry) \
                avi_entry_calc_timestamp((demuxer), (entry))
#define AVI_ENTRY_CALC_DURATION(demuxer, entry) \
                avi_entry_calc_duration((demuxer), (entry))
/*=============================================================================
									  LOCAL VARIABLES
=============================================================================*/

static GstElementClass *parent_class = NULL;


/*=============================================================================
						LOCAL FUNCTION PROTOTYPES
=============================================================================*/

GST_DEBUG_CATEGORY_STATIC(mfw_gst_avidemuxer_debug);
static void mfw_gst_avidemuxer_base_init(MFW_GST_AVIDEMUX_INFO_CLASS_T *);
static void mfw_gst_avidemuxer_class_init(MFW_GST_AVIDEMUX_INFO_CLASS_T *);
static void mfw_gst_avidemuxer_init(MFW_GST_AVIDEMUX_INFO_T *);
static void mfw_gst_avidemuxer_set_property(GObject *, guint,
					    const GValue *, GParamSpec *);
static void mfw_gst_avidemuxer_get_property(GObject *, guint,
					    GValue *, GParamSpec *);
static gboolean mfw_gst_avidemuxer_set_caps(GstPad *, GstCaps *);
static gboolean mfw_gst_avidemuxer_sink_event(GstPad *, GstEvent *);
static gboolean mfw_gst_avidemuxer_src_event(GstPad *, GstEvent *);
static gboolean mfw_gst_avidemuxer_activate(GstPad *);
static gboolean mfw_gst_avidemuxer_activate_pull(GstPad *, gboolean);
static void mfw_gst_avidemuxer_taskfunc(MFW_GST_AVIDEMUX_INFO_T *);
static GstStateChangeReturn
mfw_gst_avidemuxer_change_state(GstElement *, GstStateChange);
static gboolean avidemuxer_fill_stream_info(MFW_GST_AVIDEMUX_INFO_T *);
static int mfw_avi_read_data(unsigned char *, int, int *, gboolean *,
			     MFW_GST_AVIDEMUX_INFO_T *);
static int mfw_avi_write_data(STD_BD_SET_T *, FSR_AVI_DEMUX_STAT_T *, int,
			      gboolean, MFW_GST_AVIDEMUX_INFO_T *);
static GstStateChangeReturn
mfw_gst_avidemuxer_parse_header(MFW_GST_AVIDEMUX_INFO_T *);

static gboolean mfw_gst_avidemuxer_handle_src_query(GstPad * pad,
						    GstQuery * query);
static GstFlowReturn
mfw_gst_avidemuxer_create_index(MFW_GST_AVIDEMUX_INFO_T *);

static gboolean mfw_gst_avidemuxer_parse_index(GstElement *, GstBuffer *);
static gst_avi_index_entry *mfw_avi_demux_parse_seek(MFW_GST_AVIDEMUX_INFO_T
						    *, gint, guint64,
						    guint32);
static GstClockTime mfw_avi_demux_get_duration(MFW_GST_AVIDEMUX_INFO_T *,gint);
static gboolean mfw_gst_avidemuxer_seek(MFW_GST_AVIDEMUX_INFO_T *,
					GstPad *, GstEvent *);
static void mfw_gst_avidemuxer_reset(MFW_GST_AVIDEMUX_INFO_T *);
static GstStateChangeReturn
mfw_gst_avidemuxer_parser_init(MFW_GST_AVIDEMUX_INFO_T *);
static gboolean mfw_gst_avidemuxer_calc_timestamp(MFW_GST_AVIDEMUX_INFO_T * avi,
        gst_avi_index_entry * entry,
        guint64 * timestamp, guint64 * duration);

static guint64 avi_entry_calc_timestamp(MFW_GST_AVIDEMUX_INFO_T * avi,
        gst_avi_index_entry * entry);
static guint64 avi_entry_calc_duration(MFW_GST_AVIDEMUX_INFO_T * avi,
        gst_avi_index_entry * entry);
static void *mfw_avi_demux_get_keyframe(MFW_GST_AVIDEMUX_INFO_T * avi);

/*=============================================================================
							GLOBAL VARIABLES
=============================================================================*/
/* None. */
/*=============================================================================
							LOCAL FUNCTIONS
=============================================================================*/



/*=============================================================================
FUNCTION:   		mfw_gst_avidemuxer_reset

DESCRIPTION:		Resets the demuxer .

ARGUMENTS PASSED:	avi - The main structure of avi parser plugin context.


RETURN VALUE:	    None

PRE-CONDITIONS:     None

POST-CONDITIONS:    None

IMPORTANT NOTES:    None
=============================================================================*/

static void mfw_gst_avidemuxer_reset(MFW_GST_AVIDEMUX_INFO_T * avi)
{

    gint i;

    for (i = 0; i < avi->num_streams; i++) {
	if (avi->avi_stream[i].name)
	    g_free(avi->avi_stream[i].name);
	if (avi->avi_stream[i].src_pad)
	    gst_element_remove_pad(GST_ELEMENT(avi),
				   avi->avi_stream[i].src_pad);
    }
    memset(&avi->avi_stream, 0, sizeof(avi->avi_stream));

    avi->num_streams = 0;
    avi->num_v_streams = 0;
    avi->num_a_streams = 0;

    avi->offset = 0;

    if (avi->index_entries != NULL)
        g_free(avi->index_entries);
    avi->index_entries = NULL;

    avi->index_size = 0;
    avi->index_offset = 0;
    avi->current_entry = 0;
    g_free(avi->avih);
    avi->avih = NULL;

    if (avi->seek_event) {
	gst_event_unref(avi->seek_event);
	avi->seek_event = NULL;
    }
    avi->segment_rate = 1.0;
    avi->segment_flags = 0;
    avi->segment_start = -1;
    avi->segment_stop = -1;
}

/*=============================================================================
FUNCTION:   		avidemuxer_fill_stream_info

DESCRIPTION:		Creates the audio / video pads for and sets the 
					corresponding capabilities.

ARGUMENTS PASSED:	demuxer - The main structure of avi parser plugin context.


RETURN VALUE:		TRUE    -   Success
                    FALSE   -   failure

PRE-CONDITIONS:     None

POST-CONDITIONS:    None

IMPORTANT NOTES:    None
=============================================================================*/
static gboolean
avidemuxer_fill_stream_info(MFW_GST_AVIDEMUX_INFO_T * demuxer)
{
    guint count = 0;
    avi_stream_context *stream_ptr = NULL;
    gchar *codec_name = NULL;
    gchar *padname = NULL;
    GstPadTemplate *templ = NULL;
    GstCaps *caps = NULL;
    GstPad *pad = NULL;
    gchar *tag_name = NULL;
    FSR_AVI_DEMUX_STAT_T *stStat = NULL;
    gchar *text_msg=NULL;
    GstStructure *structure = NULL;
    const gchar *mime = NULL;
    GstTagList *list_tag = gst_tag_list_new();
    GstElementClass *klass = GST_ELEMENT_GET_CLASS(demuxer);

    GST_DEBUG(" in  gboolean avidemuxer_fill_stream_info \n");

    stStat = &demuxer->avi_str_content.stStat;

    for (count = 0; count < demuxer->num_streams; count++) {
	caps = NULL;
	stream_ptr = &(demuxer->avi_stream[count]);

	stream_ptr->strh.fcc_handler =
	    stStat->specData.stream_index[count].strh.fccHandler;
	stream_ptr->strh.flags =
	    stStat->specData.stream_index[count].strh.dwFlags;
	stream_ptr->strh.priority =
	    stStat->specData.stream_index[count].strh.wPriority;
	stream_ptr->strh.type =
	    stStat->specData.stream_index[count].strh.fccType;
	stream_ptr->strh.init_frames =
	    stStat->specData.stream_index[count].strh.dwInitialFrames;
	stream_ptr->strh.scale =
	    stStat->specData.stream_index[count].strh.dwScale;
	stream_ptr->strh.rate =
	    stStat->specData.stream_index[count].strh.dwRate;
	stream_ptr->strh.start =
	    stStat->specData.stream_index[count].strh.dwStart;
	stream_ptr->strh.length =
	    stStat->specData.stream_index[count].strh.dwLength;
	stream_ptr->strh.bufsize =
	    stStat->specData.stream_index[count].strh.
	    dwSuggestedBufferSize;
	stream_ptr->strh.quality =
	    stStat->specData.stream_index[count].strh.dwQuality;
	stream_ptr->strh.samplesize =
	    stStat->specData.stream_index[count].strh.dwSampleSize;


	switch (stream_ptr->strh.type) {
	case (GST_MAKE_FOURCC('v', 'i', 'd', 's')):
	    {
		stream_ptr->strf.vids.size = 0;
		stream_ptr->strf.vids.width = stStat->specData.video_width;
		stream_ptr->strf.vids.height =
		    stStat->specData.video_height;
		stream_ptr->strf.vids.planes = 0;
		stream_ptr->strf.vids.bit_cnt = 0;
		stream_ptr->strf.vids.compression = stStat->specData.video_compression;
		stream_ptr->strf.vids.image_size = 0;
		stream_ptr->strf.vids.xpels_meter = 0;
		stream_ptr->strf.vids.ypels_meter = 0;
		stream_ptr->strf.vids.num_colors = 0;
		stream_ptr->strf.vids.imp_colors = 0;
	    }
	    break;

	case (GST_MAKE_FOURCC('a', 'u', 'd', 's')):
	    {
		stream_ptr->strf.auds.format = stStat->specData.format_tag;
		stream_ptr->strf.auds.channels =
		    stStat->specData.audio_channels;
		stream_ptr->strf.auds.rate = stStat->specData.audio_freq;
		stream_ptr->strf.auds.av_bps =
		    stStat->specData.audio_bitrate;
		stream_ptr->strf.auds.blockalign =
		    stStat->specData.audio_rate;
		stream_ptr->strf.auds.size = 0;
	    }
	    break;

	default:
	    GST_DEBUG("\n No fourcc type is found for the stream  \n");
	    break;

	}

	switch (stream_ptr->strh.type) {
	case (GST_MAKE_FOURCC('v', 'i', 'd', 's')):
	{
		guint32 fourcc = stream_ptr->strh.fcc_handler;
                gchar *fourccname =(gchar *)(&stream_ptr->strh.fcc_handler);

                if(strncasecmp(fourccname, "AVC1", 4) == 0)
                    fourcc = GST_MAKE_FOURCC('h','2','6','4');
                else if(strncasecmp(fourccname, "MP4V", 4) == 0)
                    fourcc = GST_MAKE_FOURCC('M','P','E','G');

		padname =
		    g_strdup_printf("video_%02d", demuxer->num_v_streams);
		templ =
		    gst_element_class_get_pad_template(klass,
						       "video_%02d");
		caps =
		    gst_riff_create_video_caps(fourcc, &stream_ptr->strh,
					       &stream_ptr->strf.vids,
					       NULL, NULL, &codec_name);
      
         
        if (caps)
        {   

            structure = gst_caps_get_structure(caps, 0);

            mime = gst_structure_get_name(structure);

            if((strcmp(mime,"video/x-h264") != 0) &&
                (strcmp(mime,"video/mpeg")   != 0) &&
                (strcmp(mime,"image/jpeg")   != 0) &&
                (strcmp(mime,"video/x-h263") != 0))
            {
                GST_ELEMENT_ERROR (demuxer, STREAM, FAILED,
                    (("Internal data stream error.")),
                    ("streaming stopped, reason: "
                    "video type of mime type %s not supported ", 
                    mime));
                return FALSE;
            }

            /* If it is the video/mpeg and 
               mpegversion=(int)1;
               Change the mpegversion=(int)4.
            */
               
            if ( (!strcmp(mime,"video/mpeg")) && 
                (g_value_get_int(gst_structure_get_value(structure, "mpegversion")) == 1) 
               )
            {
#if 1            
                GValue mpegversion = { G_TYPE_INT, 4};
                gst_structure_set_value(structure,"mpegversion",&mpegversion);
#else
                gst_structure_set_name(structure,"video/mp2v");
#endif
                {
                    gchar *tmp;
                    tmp = gst_caps_to_string(caps);
                    GST_WARNING("Change AVI video caps to: %s.\n",tmp);
                    g_free(tmp);
                }                



            }
            g_print("Codec name is %s\n",codec_name);
        }      
        else
        {
            GST_ELEMENT_ERROR (demuxer, STREAM, FAILED,
                (("Internal data stream error.")),
                ("streaming stopped, reason: "
                "video type of mime type not found "));
                return FALSE;
        }


        if (codec_name)
        {  
		/* if the decoder type is mpeg4 or h263 */
        if((strcmp(codec_name,"MPEG-1 video")==0)||(strcmp(codec_name,"ITU H.26n")==0))
		{
			/* check for the supported video resolution */
			if((stream_ptr->strf.vids.width>MAX_RESOLUTION_WIDTH)||
				(stream_ptr->strf.vids.width<MIN_RESOLUTION_WIDTH)||
				(stream_ptr->strf.vids.height>MAX_RESOLUTION_HEIGHT)||
				(stream_ptr->strf.vids.height<MIN_RESOLUTION_HEIGHT))
			{
				text_msg=" Unsupported video resolution ";
			    goto Err_Msg;
			}
		}
        
		/* if the decoder type is h264 */
		if(strcmp(codec_name,"ITU H.264")==0)
		{
			/* check for the supported video resolution */
			if((stream_ptr->strf.vids.width>MAX_RESOLUTION_WIDTH)||
				(stream_ptr->strf.vids.width<MIN_RESOLUTION_WIDTH)||
				(stream_ptr->strf.vids.height>MAX_RESOLUTION_HEIGHT)||
				(stream_ptr->strf.vids.height<MIN_RESOLUTION_HEIGHT))
			{
				text_msg=" Unsupported video resolution ";
			    goto Err_Msg;
			}

		}
         }
		if (!caps) {

		    caps =
			gst_caps_new_simple("video/x-avi-unknown",
					    "fourcc", GST_TYPE_FOURCC,
					    fourcc, NULL);
		}
		tag_name = GST_TAG_VIDEO_CODEC;

		demuxer->num_v_streams++;

		break;
	    }

	case (GST_MAKE_FOURCC('a', 'u', 'd', 's')):
	    {

        /* if the number of chanels are zero */
        if(stream_ptr->strf.auds.channels==0)
            stream_ptr->strf.auds.channels = 2; /*set to default channel number*/

		/* if the audio sampling frequency is zero */
		if(stream_ptr->strf.auds.rate==0)
            stream_ptr->strf.auds.rate == 44100; /*set to default sample rate*/

		/* if the audio bit rate is zero */
		if(stream_ptr->strf.auds.av_bps==0)
            stream_ptr->strf.auds.av_bps = 128000; /*set to default bitrate*/

        if(WAVE_FORMAT_ALAW == stream_ptr->strf.auds.format)
        {
            stream_ptr->src_pad = NULL; 
            GST_DEBUG("ALAW audio, contine to next stream\n");

            /* Post message to application */
            {
                GstMessage *message = NULL;
                GError *gerror = NULL;

                gerror = g_error_new_literal(GST_STREAM_ERROR, GST_STREAM_ERROR_WRONG_TYPE, "Unsupported Audio");
                message =
                    gst_message_new_error(GST_OBJECT(GST_ELEMENT(demuxer)), gerror,
                            "debug none");
                gst_element_post_message(GST_ELEMENT(demuxer), message);
                g_error_free(gerror);
            }
	        continue;        
        }
		padname =
		    g_strdup_printf("audio_%02d", demuxer->num_a_streams);
		templ =
		    gst_element_class_get_pad_template(klass,
						       "audio_%02d");
		caps =
		    gst_riff_create_audio_caps(stream_ptr->strf.auds.
					       format, &stream_ptr->strh,
					       &stream_ptr->strf.auds,
					       NULL, NULL, &codec_name);
		if (!caps) {
		    caps =
			gst_caps_new_simple("audio/x-avi-unknown",
					    "codec_id", G_TYPE_INT,
					    stream_ptr->strf.auds.format,
					    NULL);
		}
		tag_name = GST_TAG_AUDIO_CODEC;

		demuxer->num_a_streams++;

		break;
	    }

	default:
	    {
		GST_DEBUG
		    (" \n No fourcc type is found for the stream  \n");
	    }
	    break;

	}

	/* no caps means no stream */
	if (!caps) {
	    GST_ERROR
		(" No stream detected because not able to give proper caps \n");
	    /* unref any mem that may be in use */
	    //return FALSE;
	    /* This stream has no output pad */
	    stream_ptr->src_pad = NULL;         
	    continue;
	}

	pad = stream_ptr->src_pad =
	    gst_pad_new_from_template(templ, padname);
	g_free(padname);
	gst_pad_use_fixed_caps(pad);

	/* Register the functions to the pad later */
	gst_pad_set_element_private(pad, stream_ptr);

	gst_pad_set_query_function(stream_ptr->src_pad,
				   GST_DEBUG_FUNCPTR
				   (mfw_gst_avidemuxer_handle_src_query));
	gst_pad_set_event_function(stream_ptr->src_pad,
				   GST_DEBUG_FUNCPTR
				   (mfw_gst_avidemuxer_src_event));

	gst_pad_set_caps(pad, caps);
	if (gst_pad_set_active(pad, TRUE) == TRUE) {
	    // Do nothing.
	}
	gst_element_add_pad(GST_ELEMENT(demuxer), pad);

	GST_DEBUG("Added pad %s with caps %p", GST_PAD_NAME(pad), caps);

	if (caps) {

	    gst_caps_unref(caps);
	    caps = NULL;
	}

	if (codec_name) {

	    /* creating tag list to add metadata information */
	    GstTagList *list_codec_info = gst_tag_list_new();

	    gst_tag_list_add(list_codec_info, GST_TAG_MERGE_APPEND,
			     tag_name, codec_name, NULL);

	    if (tag_name == GST_TAG_VIDEO_CODEC) {
		tag_name = MFW_GST_TAG_WIDTH;
		gst_tag_list_add(list_codec_info, GST_TAG_MERGE_APPEND,
				 tag_name, stream_ptr->strf.vids.width,
				 NULL);
		tag_name = MFW_GST_TAG_HEIGHT;
		gst_tag_list_add(list_codec_info, GST_TAG_MERGE_APPEND,
				 tag_name, stream_ptr->strf.vids.height,
				 NULL);
		tag_name = MFW_GST_TAG_FRAMERATE;
		gst_tag_list_add(list_codec_info, GST_TAG_MERGE_APPEND,
				 tag_name,
				 stStat->specData.video_framerate, NULL);

	    } else if (tag_name == GST_TAG_AUDIO_CODEC) {
		tag_name = MFW_GST_TAG_SAMPLING_FREQUENCY;
		gst_tag_list_add(list_codec_info, GST_TAG_MERGE_APPEND,
				 tag_name, stream_ptr->strf.auds.rate,
				 NULL);
		tag_name = GST_TAG_BITRATE;
		gst_tag_list_add(list_codec_info, GST_TAG_MERGE_APPEND,
				 tag_name,
				 stStat->specData.audio_bitrate * 8, NULL);

	    }

	    gst_element_found_tags_for_pad(GST_ELEMENT(demuxer), pad,
					   list_codec_info);
	    g_free(codec_name);
	    codec_name = NULL;
	}

    }

    /* creating tag list to add metadata information */
    if (stStat->metadata.title) {
	tag_name = GST_TAG_TITLE;
	gst_tag_list_add(list_tag, GST_TAG_MERGE_APPEND, tag_name,
			 stStat->metadata.title, NULL);
    }
    if (stStat->metadata.artist) {
	tag_name = GST_TAG_ARTIST;
	gst_tag_list_add(list_tag, GST_TAG_MERGE_APPEND, tag_name,
			 stStat->metadata.artist, NULL);
    }
    if (stStat->metadata.copyright) {
	tag_name = GST_TAG_COPYRIGHT;
	gst_tag_list_add(list_tag, GST_TAG_MERGE_APPEND, tag_name,
			 stStat->metadata.copyright, NULL);
    }
    if (stStat->metadata.comment) {
	tag_name = GST_TAG_COMMENT;
	gst_tag_list_add(list_tag, GST_TAG_MERGE_APPEND, tag_name,
			 stStat->metadata.comment, NULL);
    }
    if (stStat->metadata.creation) {
	tag_name = GST_TAG_DATE;
	gst_tag_list_add(list_tag, GST_TAG_MERGE_APPEND, tag_name,
			 stStat->metadata.creation, NULL);
    }
    if (stStat->metadata.language) {
	tag_name = GST_TAG_LANGUAGE_CODE;
	gst_tag_list_add(list_tag, GST_TAG_MERGE_APPEND, tag_name,
			 stStat->metadata.language, NULL);
    }
    if (stStat->metadata.source) {
	tag_name = GST_TAG_ISRC;
	gst_tag_list_add(list_tag, GST_TAG_MERGE_APPEND, tag_name,
			 stStat->metadata.source, NULL);
    }
    if (stStat->metadata.genre) {
	tag_name = GST_TAG_GENRE;
	gst_tag_list_add(list_tag, GST_TAG_MERGE_APPEND, tag_name,
			 stStat->metadata.genre, NULL);
    }
    if (stStat->metadata.product) {
	tag_name = MFW_GST_TAG_PRODUCT;
	gst_tag_list_add(list_tag, GST_TAG_MERGE_APPEND, tag_name,
			 stStat->metadata.product, NULL);
    }
    if (stStat->metadata.subject) {
	tag_name = MFW_GST_TAG_SUBJECT;
	gst_tag_list_add(list_tag, GST_TAG_MERGE_APPEND, tag_name,
			 stStat->metadata.subject, NULL);
    }
    if (stStat->metadata.software) {
	tag_name = GST_TAG_ENCODER;
	gst_tag_list_add(list_tag, GST_TAG_MERGE_APPEND, tag_name,
			 stStat->metadata.software, NULL);
    }
    gst_element_found_tags(GST_ELEMENT(demuxer), list_tag);

    GST_DEBUG(" out of  gboolean avidemuxer_fill_stream_info \n");
    return TRUE;

	/* send an error message for an unrecoverable error */
Err_Msg:
	{
    GstMessage *message = NULL;
	GError *gerror = NULL;
	gerror = g_error_new_literal(GST_STREAM_ERROR, GST_STREAM_ERROR_FORMAT, text_msg);

	message =
	    gst_message_new_error(GST_OBJECT(GST_ELEMENT(demuxer)), gerror,
				  "debug none");
	gst_element_post_message(GST_ELEMENT(demuxer), message);
	g_error_free(gerror);
	return FALSE;
	}
}

/*=============================================================================
FUNCTION:   		mfw_avi_read_data

DESCRIPTION:		This is used to fetch data from file src whenever the 
					avi demuxer needs it.

ARGUMENTS PASSED:	ptr     - Pointer to the buffer where we need data.
					length  - Length of the data required.
					input_buf_offset - Length read so far.
                    demuxer - The main structure of avi parser plugin context.
RETURN VALUE:		
					length_read - Bytes read into the buffer. 
				

PRE-CONDITIONS:     None

POST-CONDITIONS:    None

IMPORTANT NOTES:   
=============================================================================*/
static int
mfw_avi_read_data(unsigned char *pBuffer, int length,
		  int *input_buf_offset, gboolean * is_eos,
		  MFW_GST_AVIDEMUX_INFO_T * demuxer)
{

    GstBuffer *tmpbuf = NULL;
    int length_read = 0;
    GstFlowReturn ret;

    /* checking that current entry index value should not be more than total 
       enrty index if its there that means it has reached to end of file */
    if (demuxer->current_entry > demuxer->index_size - 1) {
	length_read = INVALID_READ_SIZE;
	return length_read;
    }
    ret =
	gst_pad_pull_range(demuxer->sinkpad, *input_buf_offset, length,
			   &tmpbuf);
    if (ret != GST_FLOW_OK) {
	GST_DEBUG(" FILE_READ: not able to read the data from %d\n",
		  *input_buf_offset);
	length_read = INVALID_READ_SIZE;

    } else {
	GST_DEBUG("\nRead %d bytes from the file, offset now is %d\n",
		  length, *input_buf_offset);
	length_read = GST_BUFFER_SIZE(tmpbuf);
	memcpy(pBuffer, GST_BUFFER_DATA(tmpbuf), length_read);
	*input_buf_offset += GST_BUFFER_SIZE(tmpbuf);

	if (length_read != length) {
	    GST_DEBUG("\nEnd of Stream reached. Offset now is %d\n",
		      *input_buf_offset);
	    *is_eos = TRUE;
	}
	gst_buffer_unref(tmpbuf);
    }

    return length_read;

}

/*=============================================================================
FUNCTION:   		mfw_avi_write_data

DESCRIPTION:		This is used to write the data onto respective srcpads. This function
                    is called under 2 cases. First, when we have complete chunk. In this case,
                    we need to push the data onto src pad. Second, it is called with partial
                    data. In this case, we need to just buffer the data and not send it.

ARGUMENTS PASSED:	pstOut     - Pointer to the Output buffer pointer.
					pstStat    - Pointer to the stat data structure(holds control info)
					track_type - Audio/Video.
                    to_push    - To push onto src pad or not. 
                    demuxer    - The main structure of avi parser plugin context.   
RETURN VALUE:		
					0           - Success. 
					-1          - On failure.

PRE-CONDITIONS:     None

POST-CONDITIONS:    None

IMPORTANT NOTES:   
=============================================================================*/
static int
mfw_avi_write_data(STD_BD_SET_T * pstOut, FSR_AVI_DEMUX_STAT_T * pstStat,
		   int track_type, gboolean to_push,
		   MFW_GST_AVIDEMUX_INFO_T * demuxer)
{

    STD_BD_T *temp = NULL;
    GstBuffer *src_data_buf = NULL;
    GstCaps *src_caps = NULL;
    unsigned char *tmp_buf = NULL;
    GstFlowReturn result = GST_FLOW_OK;
    gint src_buf_size = 0;
    gint i = 0;
    avi_stream_context *stream_ptr = 0;
    gint buf_count = 0;
    gst_avi_index_entry *entry = NULL;

    temp = pstOut[track_type].pPtr;

    for (i = 0; i < pstOut[track_type].usiCntBDSet; i++) {

	GST_DEBUG(" temp[%d].uliFill = %u \n", i, temp[i].uliFill);
	if (temp[i].uliFill != 0) {
	    src_buf_size += temp[i].uliFill;

	}

    }
    stream_ptr = &(demuxer->avi_stream[track_type]);
    src_caps = GST_PAD_CAPS(stream_ptr->src_pad);

    GST_DEBUG
	("\n Entered the write data function with %d complete chunk\n",
	 to_push);
    if (NULL == src_caps) {
	GST_ERROR("\n No Caps for the src pad for %d \n", track_type);
	return -1;

    }

    GST_DEBUG("\nNeed to create an output buffer of size %d for %d\n",
	      src_buf_size, track_type);
    result =
	gst_pad_alloc_buffer_and_set_caps(stream_ptr->src_pad, 0,
					  src_buf_size, src_caps,
					  &src_data_buf);

    GST_DEBUG(" result = %d\n", result);
    if (result != GST_FLOW_OK) {
	GST_ERROR("\n Cannot Create Output Buffer for %d, result is %d",
		  track_type, result);
    usleep(1);
	return -1;

    }

    tmp_buf = GST_BUFFER_DATA(src_data_buf);

    temp = pstOut[track_type].pPtr;
    for (i = 0; i < pstOut[track_type].usiCntBDSet; i++) {
	if (temp[i].uliFill != 0) {
	    memcpy(&tmp_buf[buf_count], temp[i].pPtr, temp[i].uliFill);
	    buf_count += temp[i].uliFill;
	}
    }

    /* This is the scenario when we have one complete chunk worth of data. We check for the type,
       add the time-stamp info for the chunk and push it.
       When we are pushing for the first time, we are sending new-segment as well.
     */

    if (to_push) {
	GST_DEBUG("\nPushing Data onto next element\n");

	if (demuxer->avi_stream[track_type].tmp_av_buffer != NULL
	    && src_buf_size != 0) {
	    src_data_buf =
		gst_buffer_join(demuxer->avi_stream[track_type].
				tmp_av_buffer, src_data_buf);
	}
	demuxer->avi_stream[track_type].tmp_av_buffer = NULL;

	entry = &demuxer->index_entries[demuxer->current_entry];
	GST_BUFFER_TIMESTAMP(src_data_buf) = AVI_ENTRY_CALC_TIMESTAMP(demuxer, entry);
    /* Dexter add Key frame flag */
#define AVIIF_KEYFRAME 0x10
#define GST_BUFFER_FLAG_IS_SYNC (GST_BUFFER_FLAG_LAST<<2)

    if ((entry->flags & AVIIF_KEYFRAME) == AVIIF_KEYFRAME) {
        GST_BUFFER_FLAG_SET(src_data_buf, GST_BUFFER_FLAG_IS_SYNC);
    }
 
        if (demuxer->trick_mode) {
            if (GST_CLOCK_TIME_IS_VALID(demuxer->prvBufTime)) {
                if (demuxer->trick_direction == FORWARD) {
                    gint64 tDiff = GST_BUFFER_TIMESTAMP(src_data_buf) - demuxer->prvBufTime;
                    if (tDiff > TRICK_TIME) {
                        GstEvent *new_seg;
                        gdouble rate;
                        gboolean update;
                        guint64 start;
                        guint64 stop;

                        start = demuxer->prvBufTime;
                        stop = -1;

                        rate =  tDiff / TRICK_TIME * demuxer->segment_rate;
                        new_seg = gst_event_new_new_segment
                            (TRUE, rate, GST_FORMAT_TIME,
                             start, stop, demuxer->prvBufTime);

                        gst_pad_push_event(stream_ptr->src_pad, new_seg);
                    }
                    else {
                        gst_buffer_unref(src_data_buf);
                        return TRUE;
                    }
                }
                else {
                    gint64 tDiff = demuxer->prvBufTime - GST_BUFFER_TIMESTAMP(src_data_buf) ;
                    if (tDiff > TRICK_TIME) {
                        GstEvent *new_seg;
                        gdouble rate;
                        gboolean update;
                        guint64 start;
                        guint64 stop;

                        start = GST_BUFFER_TIMESTAMP(src_data_buf);
                        stop = demuxer->prvBufTime;

                        rate =  tDiff / TRICK_TIME * demuxer->segment_rate;
                        new_seg = gst_event_new_new_segment
                            (FALSE, rate, GST_FORMAT_TIME,
                             start, stop, start);

                        gst_pad_push_event(stream_ptr->src_pad, new_seg);
                    }
                    else {
                        gst_buffer_unref(src_data_buf);
                        return TRUE;
                    }
                }
            }
            demuxer->prvBufTime = GST_BUFFER_TIMESTAMP(src_data_buf);
        }

	if (demuxer->new_segment[track_type]) {
	    guint64 start = 0;
	    demuxer->new_segment[track_type] = FALSE;
	    if (!gst_pad_push_event
		(stream_ptr->src_pad,
		 gst_event_new_new_segment(FALSE, 1.0, GST_FORMAT_TIME,
					   start, GST_CLOCK_TIME_NONE,
					   start))) {
		GST_ERROR("\nCannot send new segment to the src pad\n");
		gst_buffer_unref(src_data_buf);
		return -1;
	    }
	}


	if (src_buf_size) {
	    result = gst_pad_push(stream_ptr->src_pad, src_data_buf);

        demuxer->videosent = TRUE;
        
	} else {
	    gst_buffer_unref(src_data_buf);
	}
	if (result != GST_FLOW_OK) {
	    GST_ERROR("\n Cannot Push Data onto next element");
        usleep(1);
	    return -1;

	}
    }
    /*  This is the scenario when there is partial output. We cannot push this buffer onto src pad.
       Hence, we are accumulating that data in a buffer and then pushing it.
     */
    else {

	GST_DEBUG("\nNot Pushing Data, just writing to the buffer\n");
	if (demuxer->avi_stream[track_type].tmp_av_buffer == NULL) {
	    demuxer->avi_stream[track_type].tmp_av_buffer = src_data_buf;
	} else {

	    if (GST_BUFFER_SIZE
		(demuxer->avi_stream[track_type].tmp_av_buffer))
		demuxer->avi_stream[track_type].tmp_av_buffer =
		    gst_buffer_join(demuxer->avi_stream[track_type].
				    tmp_av_buffer, src_data_buf);
	}
    }


    return 0;
}

/*=============================================================================
FUNCTION:   	mfw_gst_avidemuxer_parse_header
	
DESCRIPTION:	
                This function parses the header information.

ARGUMENTS PASSED:	
                demuxer 	- pointer to the avi demuxer element.
				
	 
RETURN VALUE:
		        GST_STATE_CHANGE_SUCCESS    -   success.
                GST_STATE_CHANGE_FAILURE    -   failure.

PRE-CONDITIONS:
		None

POST-CONDITIONS:
		None

IMPORTANT NOTES:
		None
=============================================================================*/
static GstStateChangeReturn
mfw_gst_avidemuxer_parse_header(MFW_GST_AVIDEMUX_INFO_T * demuxer)
{


    GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;

    STD_BD_SET_T *stIn = NULL;
    STD_BD_SET_T *stOut = NULL;
    FSR_AVI_DEMUX_CTRL_T *stCtrl = NULL;
    gint i = 0;
    FSR_AVI_DEMUX_STAT_T *stStat = NULL;
    STD_RETURN_T uliResult = 0xFFFF;	//arbitrary value
    gboolean ret_val = GST_FLOW_OK;

    stOut = demuxer->avi_str_content.stOut;
    stIn = &demuxer->avi_str_content.stIn;
    stCtrl = &demuxer->avi_str_content.stCtrl;
    stStat = &demuxer->avi_str_content.stStat;

    /* ============================================================================ */
    /* Process the AVI header and return streams information                       */
    /* ============================================================================ */
    /*  no skipping in the bitstream, the AVI RIFF header is only located at the 
       beginning of an AVI file */

    if (demuxer->init_parser == FALSE) {
	GST_DEBUG
	    (" ---------------- coming in init parser ------------------- \n");

	for (i = 0; i < stIn->usiCntBDSet; i++) {	/*stIn.usiCntBDSet = FSR_AVI_DEMUX_INPUT_BUFFERS */
	    stIn->pPtr[i].pPtr =
		(void *) ((unsigned char *) demuxer->input_buf +
			  i * (AVI_DEMUX_INPUT_BUFFER_SIZE /
			       AVI_DEMUX_INPUT_BUFFERS));

	    stIn->pPtr[i].uliLen =
		AVI_DEMUX_INPUT_BUFFER_SIZE / AVI_DEMUX_INPUT_BUFFERS;

	    stIn->pPtr[i].uliFill =
		mfw_avi_read_data(stIn->pPtr[i].pPtr, stIn->pPtr[i].uliLen,
				  &demuxer->input_buf_offset,
				  &demuxer->is_eos, demuxer);


	    if (0 == stIn->pPtr[i].uliFill) {
		GST_DEBUG(" 0 bytes has been read \n");
	    }

	    demuxer->buf_filled++;
	}

	GST_DEBUG("\nGoing to parse the AVI Header now\n");
	uliResult = FSR_AVI_DEMUX_ProcessHDR(stIn, stOut, stCtrl, stStat);

	GST_DEBUG("\nParsed the AVI Header, the result is %d\n",
		  uliResult);

	demuxer->num_streams = stStat->specData.noStreams;

	switch (uliResult) {
	case STD_RETURN_SUCCESS:
	    {
		break;		/*successfull processing..move on */
	    }
	case STD_RETURN_MORED:
	    {
		GST_ERROR
		    ("\nNeed at least 12 bytes to read the AVI header\n");
		/*fill buffers before calling, this branch should never be taken */

	    }
	case STD_RETURN_ILLOPM:
	    {
		GST_ERROR
		    ("\nAVI RIFF header not found. This is not an avi file! \n");
		return GST_STATE_CHANGE_FAILURE;

	    }
	default:
	    {
		GST_ERROR
		    ("\nUnexpected return from FSR_AVI_DEMUX_ProcessHDR \n");
		/*this branch should never be taken */
		return GST_STATE_CHANGE_FAILURE;
	    }
	}

    }
    uliResult = 0xFFFF;		/*reset value to arbitrary */

	/*----- Print the AVI header information --------------------------------*/
    GST_DEBUG("Number of streams : %u\n", stStat->specData.noStreams);
    GST_DEBUG("\tMicro seconds per frame : %u\n",
	      stStat->specData.video_freq);
    GST_DEBUG("\tFrame width             : %u\n",
	      stStat->specData.video_width);
    GST_DEBUG("\tFrame height            : %u\n",
	      stStat->specData.video_height);
    GST_DEBUG("\tFrame Rate              : %f\n",
	      stStat->specData.video_framerate);
    GST_DEBUG("\n\tAudio Format [MP3:1]  : %d\n",
	      stStat->specData.audio_dec);
    GST_DEBUG("\tAudio frequency         : %d\n",
	      stStat->specData.audio_freq);
    GST_DEBUG("\tAudio BitRate           : %d\n",
	      stStat->specData.audio_bitrate);
    GST_DEBUG("\tAudio channels          : %d\n",
	      stStat->specData.audio_channels);
    GST_DEBUG("\tAudio [CBR:1, VBR:0]    : %d\n",
	      stStat->specData.audio_rate);

	/**** Setting Pads for the streams present ****/
    ret_val = avidemuxer_fill_stream_info(demuxer);
    if (FALSE == ret_val) {
	GST_ERROR
	    (" not able to fill stream information this means this is not a valid stream\ 
                    so going out of demuxer . \n");
	return GST_STATE_CHANGE_FAILURE;

    }

	/* code added for sending msgs to application. */
	GST_DEBUG("signaling no more pads from mfw_avidemuxer");
	gst_element_no_more_pads (GST_ELEMENT (demuxer));

    demuxer->offset = (guint64) (stStat->uliRewFillIn) - 12;

    //create index table
    if (GST_FLOW_OK != mfw_gst_avidemuxer_create_index(demuxer)) {
	GST_ERROR("mfw_gst_avidemuxer_create_index is error");
	return GST_STATE_CHANGE_FAILURE;
    }
#ifdef MFW_AVI_TEST_SEEK
{
    gst_avi_index_entry *entry;
    guint64 seek_time =100000000000;    
    entry =	mfw_avi_demux_parse_seek(demuxer, 0, seek_time,
					(guint32) GST_RIFF_IF_KEYFRAME);
    demuxer->current_entry = entry->index_nr;   
    GST_DEBUG("cur entry %d\n", demuxer->current_entry);
    //demuxer->segment_start =seek_time;
    return GST_STATE_CHANGE_FAILURE; //Test only, fail the state change.
}

#endif
    return ret;

}

/*=============================================================================
FUNCTION:   		mfw_gst_avidemuxer_create_index

DESCRIPTION:		This function creates  a table which contains the entries for 
                    keyframes.


ARGUMENTS PASSED:	avi - The main structure of avi parser plugin context.


RETURN VALUE:	    GST_FLOW_OK -   success 

PRE-CONDITIONS:     None

POST-CONDITIONS:    None

IMPORTANT NOTES:    None
=============================================================================*/
static GstFlowReturn
mfw_gst_avidemuxer_create_index(MFW_GST_AVIDEMUX_INFO_T * avi)
{

    GstFlowReturn res = GST_FLOW_OK;
    gboolean ret = FALSE;
    guint32 tag = 0;
    guint32 size = 0;
    GList *index = NULL, *alloc = NULL;
    guint64 offset = (guint64)(avi->offset);
    GstBuffer *buf = NULL;
    gst_avi_index_entry *entry = NULL;
    guint64 ts=0, dur=0;
    gint i;    

    GST_DEBUG("into mfw_gst_avidemuxer_create_index with offset = %d", offset);
    while(1)
    {
    /* get position */
    if (gst_pad_pull_range(avi->sinkpad, offset, MFW_AVI_HEADER_SIZE, &buf) != GST_FLOW_OK)
    {
        return GST_FLOW_ERROR;
    }
    else if (GST_BUFFER_SIZE (buf) < MFW_AVI_HEADER_SIZE)
    {
        return GST_FLOW_ERROR;
    }

    /* check tag first before blindy trying to read 'size' bytes */
    tag = GST_READ_UINT32_LE (GST_BUFFER_DATA (buf));
    size = GST_READ_UINT32_LE (GST_BUFFER_DATA (buf) + 4);
        if(tag == GST_MAKE_FOURCC('L', 'I', 'S', 'T'))
        {
            GST_DEBUG("movi list got\n");
            break;
        }
        GST_DEBUG("Non movi list found! tag 0x%x, size %d\n", tag, size);        
        avi->offset += MFW_AVI_HEADER_SIZE + ((size + 1) & ~1);
        offset = (guint64)(avi->offset);
    }   
    GST_DEBUG("movi list offset %ld\n", avi->offset);    
    offset += MFW_AVI_HEADER_SIZE + ((size + 1) & ~1);

    gst_buffer_unref(buf);
    GST_DEBUG("\n Found idx1 offset at %lld\n", offset);

    /* Read IDX1 chunk data into 'buf', 'offset' is updated to the end of IDX1 chunk */
    res = gst_riff_read_chunk(GST_ELEMENT(avi), avi->sinkpad, &offset, &tag, &buf);
    if ( res != GST_FLOW_OK)
        return GST_FLOW_ERROR;
    else if (tag != GST_RIFF_TAG_idx1) {
        GST_ERROR("Idx1 offset error\n");
        gst_buffer_unref(buf);
        return GST_FLOW_ERROR;
    }    

   ret = mfw_gst_avidemuxer_parse_index(GST_ELEMENT(avi), buf);
   if (!ret)
      return GST_FLOW_ERROR;
   
    // send initial discont
    avi->segment_start = 0;
    entry = &avi->index_entries[avi->index_size-1];
    mfw_gst_avidemuxer_calc_timestamp(avi, entry, &ts, &dur);
    avi->total_duration = ts;
    avi->segment_stop = ts + dur; 

    // get video duration
    for (i = avi->index_size-1; i >= 0;  i--) {
        entry = &avi->index_entries[i];
        if (entry->stream_nr == AVI_DEMUX_VIDEO_STREAM) {
            break;  
        }
    }
    mfw_gst_avidemuxer_calc_timestamp(avi, entry, &ts, &dur);
    avi->max_video_time = ts + dur;

    // get audio duration
    for (i = avi->index_size-1; i >= 0;  i--) {
        entry = &avi->index_entries[i];
        if (entry->stream_nr == AVI_DEMUX_AUDIO_STREAM) {
            break;  
        }
    }
    mfw_gst_avidemuxer_calc_timestamp(avi, entry, &ts, &dur);
    avi->max_audio_time = ts + dur;
    GST_DEBUG("Out mfw_gst_avidemuxer_create_index");

    return GST_FLOW_OK;
}

/*=============================================================================
FUNCTION:   		mfw_gst_avidemuxer_parse_index

DESCRIPTION:		This function parses the stream to get keyframe.

ARGUMENTS PASSED:	avi - The main structure of avi parser plugin context.


RETURN VALUE:	    None

PRE-CONDITIONS:     None

POST-CONDITIONS:    None

IMPORTANT NOTES:    None
=============================================================================*/
static gboolean mfw_gst_avidemuxer_parse_index(GstElement *element, GstBuffer * buf)
{

    MFW_GST_AVIDEMUX_INFO_T *avi = MFW_GST_AVI_DEMUXER(element);
    guint32 pos_before = avi->offset; /* start offset of 'movi' list, point to list tag 'LIST' */
    gst_avi_index_entry *target_entry = NULL;
    guint8 *data;
    guint i,n;
    guint num; /* number of index entries in 'idx1' */
    gchar *chunkid;
    gboolean is_drm_entry = FALSE; /*Whether current entry is a drm entry */
    guint resn;

    GST_DEBUG("into mfw_gst_avidemuxer_new_parse_index\n");

    if (!buf) {
	return FALSE;
    }

    data = GST_BUFFER_DATA(buf);
    num = GST_BUFFER_SIZE(buf) / sizeof(gst_riff_index_entry);

    if (num > MAX_INDEX_ENTRY_COUNT)
    {
       GST_ERROR("Error if index entry is too large.\n");
       return FALSE;   
    }

    avi->index_entries = g_new(gst_avi_index_entry, num);
   
    if (avi->index_entries == NULL)
   {
    GST_ERROR("\nError in allocating memory for index_entries\n");
    return FALSE;
   } 
    GST_DEBUG("entry size=%d,num=%d\n",sizeof(gst_avi_index_entry),num);

    for (i = 0, n = 0; i < num; i++) {
	gst_riff_index_entry riff_entry, *_entry;
	avi_stream_context *stream;
	gint stream_nr;

        target_entry = &(avi->index_entries[n]);
 
	_entry = &((gst_riff_index_entry *) data)[i];
	riff_entry.id = GUINT32_FROM_LE(_entry->id);
	riff_entry.offset = GUINT32_FROM_LE(_entry->offset);
	riff_entry.flags = GUINT32_FROM_LE(_entry->flags);
	riff_entry.size = GUINT32_FROM_LE(_entry->size);
	//printf("idx %d, flag %ld, size %d\n", i, riff_entry.flags, riff_entry.size);
  
	if (riff_entry.id == GST_RIFF_rec || riff_entry.id == 0 ||
            riff_entry.id == GST_RIFF_TAG_JUNK || 
	    (riff_entry.offset == 0 && n > 0))
	    continue;
    for (resn=0; resn<10; resn++)
        if (riff_entry.id == GST_MAKE_FOURCC ('R', 'E', 'S', '0' + resn))
            continue;
        
    is_drm_entry = FALSE;

	chunkid = (gchar *) (&riff_entry.id);
    if ((strncasecmp(chunkid, "00dd", 4) == 0) ||
        (strncasecmp(chunkid, "01dd", 4) == 0))

    {   /* Quit parsing if DRM is implemented on the stream */
              g_print("Not support DRM encrpted .\n");
              return FALSE;
    }    

	if (strncasecmp(chunkid, "00", 2) == 0)
	    stream_nr = 0;
	else if (strncasecmp(chunkid, "01", 2) == 0)
	    stream_nr = 1;
    else
        continue;

    /* Modify offset from list type 'movi' to list tag 'LIST' */
    target_entry->offset = riff_entry.offset+8;
    if(is_drm_entry)
    {   /* Make sure this entry will not be take as a key frame,
        riff_entry.flags &= (~AVIIF_KEYFRAME);
        riff_entry.flags |= AVIIF_NOTIME; /* For double check in seek() */        
    }        
        target_entry->flags = riff_entry.flags;
        target_entry->size = riff_entry.size;

	target_entry->stream_nr = stream_nr;
	stream = &avi->avi_stream[stream_nr];
 
        stream->num = stream_nr;
 
	// figure out if the index is 0 based or relative to the MOVI start 
	if (n == 0) {
        
	    //if (target_entry->offset < pos_before)
		avi->index_offset = pos_before + 8; /* ENGR66591 */
	    //else
		//avi->index_offset = 0;
	}
 
	if (stream->strh.type == GST_RIFF_FCC_auds) {

	    // all audio frames are keyframes
	    target_entry->flags |= GST_RIFF_IF_KEYFRAME;
	    // constant rate stream
           
            gboolean is_vbr = (stream->strh.samplesize == 0)&&(stream->strh.scale > 1);
            if (!is_vbr) {
		target_entry->frames_before = (guint32)(stream->total_bytes);
	    }
	    //VBR
	    else {
		target_entry->frames_before = stream->total_frames;
	    }
	} else {
	    /* Video entry or DRM entry */
		target_entry->frames_before = stream->total_frames;
	}
    if(!is_drm_entry)
    {   /* Only accumulate time and data for non-drm a/v entries */
	    stream->total_bytes += target_entry->size;
	    stream->total_frames++;
    }
	n++;
    }
    
    avi->index_size = n;
    gst_buffer_unref(buf);
    
    GST_DEBUG("out mfw_gst_avidemuxer_parse_index"); 
    return TRUE;
}					   

/*=============================================================================
FUNCTION:			mfw_gst_avidemuxer_taskfunc

DESCRIPTION:		This is the main functions, where calls to AVI Parser 
					are made.

ARGUMENTS PASSED:	demuxer - The context of the main avi demuxer element.


RETURN VALUE:		None

PRE-CONDITIONS:     None

POST-CONDITIONS:	None

IMPORTANT NOTES:    None
=============================================================================*/
static void mfw_gst_avidemuxer_taskfunc(MFW_GST_AVIDEMUX_INFO_T * demuxer)
{

    STD_BD_SET_T *stIn = NULL;
    STD_BD_SET_T *stOut = NULL;
    FSR_AVI_DEMUX_CTRL_T *stCtrl = NULL;
    gint i = 0;
    FSR_AVI_DEMUX_STAT_T *stStat = NULL;
    STD_RETURN_T uliResult = 0xFFFF;	//arbitrary value
    gint j = 0;
    UWord16 save_indx = 0;
    gst_avi_index_entry *entry = NULL;
    GST_DEBUG(" in mfw_gst_avidemuxer_taskfunc function\n");

    stOut = demuxer->avi_str_content.stOut;
    stIn = &demuxer->avi_str_content.stIn;
    stCtrl = &demuxer->avi_str_content.stCtrl;
    stStat = &demuxer->avi_str_content.stStat;

    /* ======================================================================= */
    /* Process in a loop the entire bitstream. Return from process when either all */
    /* output buffers were filled or more input data is needed.                    */
    /* ======================================================================== */
    /*
     *  The output buffers are "shared" with the input buffers
     *  No copying is needed this way, but the framework should be aware of this 
     */
    if (demuxer->isWork) {
	gint i;

	stStat->usiRewBDIn = 0;
	stStat->uliRewFillIn = 0;

	entry = &demuxer->index_entries[demuxer->current_entry];
	stIn->pPtr[0].uliLen = entry->size + MFW_AVI_HEADER_SIZE;
	demuxer->input_buf_offset =
	entry->offset + demuxer->index_offset - MFW_AVI_HEADER_SIZE;

	stIn->pPtr[0].uliFill =
	mfw_avi_read_data(stIn->pPtr[0].pPtr, stIn->pPtr[0].uliLen,
			      &demuxer->input_buf_offset,
			      &demuxer->is_eos, demuxer);

	stIn->pPtr[1].uliFill = 0;
	demuxer->buf_filled++;
	demuxer->last_index_read = 0;
	demuxer->isWork = FALSE;
    }

    if (demuxer->seek_event) {
	gint i;
	GST_DEBUG("mfw_avidemux:seek is true");
	for (i = 0; i < demuxer->num_streams; i++) {
	    avi_stream_context *stream = &(demuxer->avi_stream[i]);
        if(stream->src_pad)
        {
	        gst_event_ref(demuxer->seek_event);
	        gst_pad_push_event(stream->src_pad, demuxer->seek_event);
        }
	}
	gst_event_unref(demuxer->seek_event);
	demuxer->seek_event = NULL;
    if (demuxer->audio_eos) {
        GstFlowReturn result = GST_FLOW_OK;
        avi_stream_context *stream_ptr = NULL;
        GstEvent *event = NULL;
        
        stream_ptr = &(demuxer->avi_stream[1]);
        if(stream_ptr->src_pad)
        {
            event = gst_event_new_eos();
            result = gst_pad_push_event(stream_ptr->src_pad, event);
            if (result != TRUE) {
                GST_ERROR("\n Error in pushing the EOS event, result is %d\n",
                        result);
            }
        }
    }

    if (demuxer->video_eos) {
        GstFlowReturn result = GST_FLOW_OK;
        avi_stream_context *stream_ptr = NULL;
        GstEvent *event = NULL;
        
        stream_ptr = &(demuxer->avi_stream[0]);
        if(stream_ptr->src_pad)
        {
            event = gst_event_new_eos();
            result = gst_pad_push_event(stream_ptr->src_pad, event);
            if (result != TRUE) {
                GST_ERROR("\n Error in pushing the EOS event, result is %d\n",
                        result);

            }
        }
    }

	//directly read two new blocks from new offset
	stStat->usiRewBDIn = 0;
	stStat->uliRewFillIn = 0;
	entry = &demuxer->index_entries[demuxer->current_entry];
	stIn->pPtr[0].uliLen = entry->size + MFW_AVI_HEADER_SIZE;
	demuxer->input_buf_offset =
        entry->offset + demuxer->index_offset - MFW_AVI_HEADER_SIZE;

	stIn->pPtr[0].uliFill =
	    mfw_avi_read_data(stIn->pPtr[0].pPtr, stIn->pPtr[0].uliLen,
			      &demuxer->input_buf_offset,
			      &demuxer->is_eos, demuxer);
	stIn->pPtr[1].uliFill = 0;
	demuxer->buf_filled++;
	demuxer->last_index_read = 0;
    }

    /*
     *  The output buffers are "shared" with the input buffers
     * No copying is needed this way, but the framework should be aware of this 
     */

    for (i = 0; i < 2; i++) {
	for (j = 0; j < stOut[i].usiCntBDSet; j++) {
	    stOut[i].pPtr[j].uliFill = 0;
	    stOut[i].pPtr[j].uliLen = 0;
	}
    }
    save_indx = stStat->usiRewBDIn;

    uliResult = FSR_AVI_DEMUX_Process(stIn, stOut, stCtrl, stStat);


    switch (uliResult) {
    case STD_RETURN_AGAIN:
    case STD_RETURN_SUCCESS:

    case STD_RETURN_MORED:
	{
	    STD_BD_T *temp;
	    gst_avi_index_entry *entry;

	    GST_DEBUG("More data needed\n");
	    GST_DEBUG("At index %d and offset %d\n",
		      stStat->usiRewBDIn, stStat->uliRewFillIn);

	    entry = &demuxer->index_entries[demuxer->current_entry];

	    if (demuxer->num_v_streams && entry->stream_nr == AVI_DEMUX_VIDEO_STREAM) {
		if (-1 ==
		    mfw_avi_write_data(stOut, stStat,
				       AVI_DEMUX_VIDEO_STREAM, TRUE,
				       demuxer)) {

		    GST_DEBUG(" not able to write data \n");

		}
	    }

	    if (demuxer->num_a_streams && entry->stream_nr == AVI_DEMUX_AUDIO_STREAM) {
            guint64 ts=0, dur=0;
		if (-1 ==
		    mfw_avi_write_data(stOut, stStat,
				       AVI_DEMUX_AUDIO_STREAM, TRUE,
				       demuxer)) {

		    GST_DEBUG(" not able to write data \n");

		}
            mfw_gst_avidemuxer_calc_timestamp(demuxer, entry, &ts, &dur);
        if ((ts + dur) == demuxer->max_audio_time) {
            GstFlowReturn result = GST_FLOW_OK;
            avi_stream_context *stream_ptr = NULL;
            GstEvent *event = NULL;
            
            stream_ptr = &(demuxer->avi_stream[1]);
            if(stream_ptr->src_pad)
            {
                event = gst_event_new_eos();
                result = gst_pad_push_event(stream_ptr->src_pad, event);
                if (result != TRUE) {
                    GST_ERROR("\n Error in pushing the EOS event, result is %d\n",
                            result);
                }
            }
        }
        }
        if (demuxer->trick_direction == BACKWARD)
        {
            demuxer->current_entry--; /*backward to find the previous key frame */
        }
        else
	    demuxer->current_entry++;

           /* index read over goto end */
           if(demuxer->current_entry>=demuxer->index_size)
           {
                goto demux_end;
           }

	    /*
	     * read data in the empty input buffers 
	     */
	    if (demuxer->end_bitstream == 1) {

		goto demux_end;
	    }

	    if (stStat->usiEndBitstream == 1) {

		stStat->usiEndBitstream = 0;
		goto demux_end;
	    }

	    GST_DEBUG("Buffers filled so far %d\n", demuxer->buf_filled);
	    for (i = 0; i < stIn->usiCntBDSet; i++)
		GST_DEBUG("%d... fill %d\n", i, stIn->pPtr[i].uliFill);
        if (demuxer->trick_mode == TRUE)
	    {
            {
                mfw_avi_demux_get_keyframe (demuxer);
                if (demuxer->current_entry <= 0 
                    || demuxer->current_entry >= demuxer->index_size)
                    goto demux_end; /* we got the end of the stream */
            }
        }
		i = stStat->usiRewBDIn;
		entry = &demuxer->index_entries[demuxer->current_entry];
		stIn->pPtr[i].uliLen = entry->size + MFW_AVI_HEADER_SIZE;
		demuxer->input_buf_offset =
		entry->offset + demuxer->index_offset - MFW_AVI_HEADER_SIZE;
		demuxer->buf_filled++;
		stIn->pPtr[i].uliFill =
		    mfw_avi_read_data(stIn->pPtr[i].pPtr,
				      stIn->pPtr[i].uliLen,
				      &demuxer->input_buf_offset,
				      &demuxer->is_eos, demuxer);

        if(INVALID_READ_SIZE == stIn->pPtr[i].uliFill)
        {   /* ENGR67717:read may fail if source filter start to flush data on seeking.
            Emit eos so that this task won't be called repetitively */
           stIn->pPtr[i].uliFill = 0;
           goto demux_end;

        }            
		else if (stIn->pPtr[i].uliFill < stIn->pPtr[i].uliLen)
		    demuxer->end_bitstream = 1;

		stIn->pPtr[1 - i].uliFill = 0;
	   

	    demuxer->last_index_read = stStat->usiRewBDIn;
	    for (i = 0; i < stIn->usiCntBDSet; i++)
		GST_DEBUG("%d... fill %d\n", i, stIn->pPtr[i].uliFill);
	}
	break;
    case STD_RETURN_OORPAR:
	{

	    GstMessage *message = NULL;
	    GError *gerror = NULL;
	    GST_ERROR("Hint: Make sure the return mode is\
                       set to STD_MODE_NEED for both input and output\n");
	    gerror = g_error_new_literal(1, 0,
					 "One of the parameters given to FSR_AVI_DEMUX_Process through control is out of range!");
	    message =
		gst_message_new_error(GST_OBJECT(GST_ELEMENT(demuxer)),
				      gerror, "debug none");
	    gst_element_post_message(GST_ELEMENT(demuxer), message);

	    g_error_free(gerror);
	    break;

	}
    default:
	{
	    GST_ERROR("Not OK at all ret %d\n",uliResult);

	    goto demux_end;
	}
    }

    if (demuxer->videosent) {
        /* Keep the balance of demuxer and codec */    
        usleep(DEMUX_WAIT_INTERVAL);
        demuxer->videosent = FALSE;
    }

    return;

  demux_end:
    /* ============================================================================ */
    /* Close the Demultiplexer instance and also close all opened i/o files        */
    /* ============================================================================ */
    //FSR_AVI_DEMUX_Close(stCtrl, stStat); /* Amanda: Not close here. Otherwise internal memory becomes invalid */
    for (i = 0; i < demuxer->num_streams; i++) {
	GstFlowReturn result = GST_FLOW_OK;
	avi_stream_context *stream_ptr = NULL;
	GstEvent *event = NULL;

	stream_ptr = &(demuxer->avi_stream[i]);

        if(stream_ptr->src_pad)
        {
            event = gst_event_new_eos();
        	result = gst_pad_push_event(stream_ptr->src_pad, event);
            if (result != TRUE) {
        	    GST_ERROR("\n Error in pushing the EOS event, result is %d\n",
        		      result);
        	}
        }
    }

    gst_pad_pause_task(demuxer->sinkpad);
    return;


}

/*=============================================================================
FUNCTION:			mfw_gst_avidemuxer_activate_pull

DESCRIPTION:		This will start the demuxer task function in "pull" based	
					scheduling model.

ARGUMENTS PASSED:	pad		- sink pad, where the data will be pulled into, from	
					file source.
					active 	- 
 

RETURN VALUE:		
		TRUE      -    if operation was successful
		FALSE     -    if operation was unsuccessful
PRE-CONDITIONS:     None

POST-CONDITIONS:    None

IMPORTANT NOTES:    None
=============================================================================*/
static gboolean mfw_gst_avidemuxer_activate_pull(GstPad * pad,
						 gboolean active)
{
    gboolean ret_val = TRUE;
    MFW_GST_AVIDEMUX_INFO_T *demuxer =
	MFW_GST_AVI_DEMUXER(GST_PAD_PARENT(pad));
    if (active) {
	ret_val = gst_pad_start_task(pad, (GstTaskFunction)
				     mfw_gst_avidemuxer_taskfunc, demuxer);
	if (ret_val == FALSE) {
	    GST_ERROR("Task could not be started !!\n");
	    return FALSE;

	}
    } else {
	return gst_pad_stop_task(pad);
    }
    return TRUE;
}

/*=============================================================================
FUNCTION:   	mfw_gst_avidemuxer_activate

DESCRIPTION:    it will call gst_pad_activate_pull which activates or
				deactivates the given pad in pull mode via dispatching to the
				pad's activepullfunc

ARGUMENTS PASSED:
		 pad       -    pointer to GstPad to activate or deactivate

RETURN VALUE:
		 TRUE      -    if operation was successful
		 FALSE     -    if operation was unsuccessful

PRE-CONDITIONS:
		None

POST-CONDITIONS:
		None

IMPORTANT NOTES:
		None
=============================================================================*/
static gboolean mfw_gst_avidemuxer_activate(GstPad * pad)
{
    if (gst_pad_check_pull_range(pad)) {
	return gst_pad_activate_pull(pad, TRUE);
    } else {
	return FALSE;
    }

}

/*=============================================================================

FUNCTION:   mfw_gst_avidemuxer_handle_src_query   

DESCRIPTION:    performs query on src pad.    

ARGUMENTS PASSED:
        pad     -   pointer to GstPad
        query   -   pointer to GstQuery        
            
RETURN VALUE:
        TRUE    -   success
        FALSE   -   failure

PRE-CONDITIONS:
        None

POST-CONDITIONS:
		None

IMPORTANT NOTES:
        None

==============================================================================*/
static gboolean mfw_gst_avidemuxer_handle_src_query(GstPad * pad,
						    GstQuery * query)
{

    gboolean res = TRUE;
    MFW_GST_AVIDEMUX_INFO_T *demux =
	MFW_GST_AVI_DEMUXER(GST_PAD_PARENT(pad));
    avi_stream_context *stream = gst_pad_get_element_private(pad);

    GST_DEBUG("mfw_gst_avidemuxer_handle_src_query===%d\n",
	      GST_QUERY_TYPE(query));

    if (&(stream->strh) == NULL)
	return FALSE;

    switch (GST_QUERY_TYPE(query)) {
    case GST_QUERY_DURATION:
	{
	    gint64 len;

            len= mfw_avi_demux_get_duration(demux,stream->num);

	    GST_DEBUG(" duration= %  " GST_TIME_FORMAT,
		      GST_TIME_ARGS(len));

	    gst_query_set_duration(query, GST_FORMAT_TIME, len);
	    break;
	}
    default:
	res = FALSE;
	break;
    }

    return res;
}

/*=============================================================================
FUNCTION:   	mfw_gst_avidemuxer_src_event

DESCRIPTION:	Handles an event on the source pad.

ARGUMENTS PASSED:
		pad        -    pointer to pad
		event      -    pointer to event
RETURN VALUE:
		TRUE       -	if event is sent to source properly
		FALSE	   -	if event is not sent to source properly

PRE-CONDITIONS:    None

POST-CONDITIONS:   None

IMPORTANT NOTES:   None
=============================================================================*/
static gboolean mfw_gst_avidemuxer_src_event(GstPad * pad,
					     GstEvent * event)
{

    gboolean res = TRUE;
    MFW_GST_AVIDEMUX_INFO_T *avi =
	MFW_GST_AVI_DEMUXER(GST_PAD_PARENT(pad));
    GstFormat format;
    GstSeekFlags flags;
    gdouble rate;
    gint64 start, stop;
    GstSeekType start_type, stop_type;
    gboolean update_start = TRUE;
    gboolean update_stop = TRUE;
    GstClockTime start_time;
    gboolean flush, keyframe;
    guint stream;
    gst_avi_index_entry *entry;
    avi_stream_context *stream_ptr = gst_pad_get_element_private(pad);
    GST_DEBUG("Into mfw_gst_avidemuxer_src_event() function \n");
    switch (GST_EVENT_TYPE(event)) {
	/* the all-formats seek logic */
    case GST_EVENT_SEEK:

	gst_event_parse_seek(event, &rate, &format, &flags, &start_type,
			     &start, &stop_type, &stop);

	GST_DEBUG_OBJECT(avi, "seek format %d, start:%lld, stop:%lld",
			 format, start, stop);

    avi->audio_eos = FALSE;
    avi->video_eos = FALSE;
        if (format != GST_FORMAT_TIME)
        {
           GST_WARNING("SEEK event not TIME format.\n");
           res = TRUE;
           goto done;
        }

	if (start >= avi->total_duration)
            start = avi->total_duration;
	
        /* now store the values */
	avi->segment_rate = rate;
	avi->segment_flags = flags;
	avi->segment_start = start;

	flush = avi->segment_flags & GST_SEEK_FLAG_FLUSH;
	keyframe = avi->segment_flags & GST_SEEK_FLAG_KEY_UNIT;

	if (flush) {

	    for (stream = avi->num_streams; stream--;) {
            if(avi->avi_stream[stream].src_pad)
            {
		        res = gst_pad_push_event(avi->avi_stream[stream].src_pad,
				       gst_event_new_flush_start());
            }
	    }
	    gst_pad_push_event(avi->sinkpad, gst_event_new_flush_start());

	} else
	    gst_pad_pause_task(avi->sinkpad);

	GST_PAD_STREAM_LOCK(avi->sinkpad);

        avi->trick_mode = FALSE;
        avi->trick_direction = 0;
        avi->prvBufTime = GST_CLOCK_TIME_NONE;

        if (rate != 1.0) {
            avi->trick_mode = TRUE;
            avi->audio_eos = TRUE;
            if (rate > 0)
                avi->trick_direction = FORWARD;
            else if (rate < 0)
                avi->trick_direction = BACKWARD;
        }
	/* fill current_entry according to flags and update */
	if (update_start) {
	    entry =
		mfw_avi_demux_parse_seek(avi, 0, avi->segment_start,
					(guint32) GST_RIFF_IF_KEYFRAME);
	    if (entry) {
		GST_DEBUG_OBJECT(avi,
				 "Got keyframe entry %d [stream:%d / ts:%"
				 GST_TIME_FORMAT " / duration:%"
				 GST_TIME_FORMAT "]", AVI_ENTRY_INDEX_NR(avi, entry),
				 entry->stream_nr,
				 GST_TIME_ARGS(AVI_ENTRY_CALC_TIMESTAMP(avi, entry)),
				 GST_TIME_ARGS(AVI_ENTRY_CALC_DURATION(avi, entry)));
		avi->current_entry = AVI_ENTRY_INDEX_NR(avi, entry);
        GST_DEBUG("cur entry idx %d\n", avi->current_entry);

	    } else {
		GST_WARNING_OBJECT(avi,
				   "Couldn't find AviIndexEntry for time:%"
				   GST_TIME_FORMAT,
				   GST_TIME_ARGS(avi->segment_start));
        avi->current_entry = 0; /* Amanda: rewind to the start of movie on seeking failure */
        avi->segment_start = 0;
        GST_DEBUG("No entry is met, rewind to start. cur entry idx %d\n", avi->current_entry);
	    }
	}

	GST_DEBUG("seek: %" GST_TIME_FORMAT " -- %" GST_TIME_FORMAT
		  " keyframe seeking:%d",
		  GST_TIME_ARGS(avi->segment_start),
		  GST_TIME_ARGS(avi->segment_stop), keyframe);

        /* If Fast forward, we need to seek to next key frame */
        if (avi->trick_direction == FORWARD) {
            if (start > AVI_ENTRY_CALC_TIMESTAMP(avi, &(avi->index_entries[avi->current_entry]))) {
                    avi->current_entry ++;
                mfw_avi_demux_get_keyframe(avi);
            }
        }
	if (keyframe)
	    start_time = AVI_ENTRY_CALC_TIMESTAMP(avi, &(avi->index_entries[avi->current_entry]));
	else
	    start_time = avi->segment_start;
    GST_DEBUG("segment start time %lld\n", start_time);

            if (avi->trick_direction == BACKWARD) {
	avi->seek_event = gst_event_new_new_segment
	    (!update_start, avi->segment_rate, GST_FORMAT_TIME,
                     start_time, -1/*start_time*/, start_time);
            }
            else {
                avi->seek_event = gst_event_new_new_segment
                    (!update_start, avi->segment_rate, GST_FORMAT_TIME,
	     start_time, avi->segment_stop, start_time);
            }
	if (flush) {
	    for (stream = avi->num_streams; stream--;) {
            if(avi->avi_stream[stream].src_pad)
            {
                res = gst_pad_push_event(avi->avi_stream[stream].src_pad,
				       gst_event_new_flush_stop());
            }		
	    }

	    gst_pad_push_event(avi->sinkpad, gst_event_new_flush_stop());
	}

	gst_pad_start_task(avi->sinkpad,
			   (GstTaskFunction) mfw_gst_avidemuxer_taskfunc,
			   avi);

	GST_PAD_STREAM_UNLOCK(avi->sinkpad);

	break;

    default:
	res = FALSE;
	break;
    }

  done:
    gst_event_unref(event);

    GST_DEBUG("Out of mfw_gst_avidemuxer_src_event() function \n");
    return res;
}

/*=============================================================================

FUNCTION:   mfw_avi_demux_parse_seek   

DESCRIPTION:    finds out the correct entry for keyframe at the time of seek.        

ARGUMENTS PASSED:
        avi         -   The main structure of avi parser plugin context.
        stream_nr   -   its 0 here , default for video .
        time        -   start time for segment.
        flags       -   GST_RIFF_IF_KEYFRAME
            
RETURN VALUE:
        entry       -   pointer to entry.  
      
PRE-CONDITIONS:
        None

POST-CONDITIONS:
		None

IMPORTANT NOTES:
        None

==============================================================================*/
static gst_avi_index_entry *mfw_avi_demux_parse_seek(MFW_GST_AVIDEMUX_INFO_T * avi, 
                            gint stream_nr,
						    guint64 time,
						    guint32 flags)
{
    gst_avi_index_entry *entry = NULL, *last_entry = NULL;
    gint i;
    guint64 ts;

    GST_DEBUG("\n____ Seek stream %d to time %lld ________________\n\n", stream_nr, time);

    if (time > avi->max_audio_time) {
        stream_nr = AVI_DEMUX_VIDEO_STREAM;
        avi->audio_eos = TRUE;
    }

    if (time > avi->max_video_time) {
        stream_nr = AVI_DEMUX_AUDIO_STREAM;
        avi->video_eos = TRUE;
    }
    
    for (i = 0; i < avi->index_size; i++) {
	entry = &avi->index_entries[i];
	if (entry->stream_nr == stream_nr) {
	    ts = AVI_ENTRY_CALC_TIMESTAMP(avi, entry);
	    if (ts > time)
		break;
	    if (ts <= time && (entry->flags & flags) == flags)
		last_entry = entry;
	}

    }
    if (last_entry == NULL) { 
     /* no key frame flag for video case, we will return first frame */
        for (i = 0; i < avi->index_size; i++) {
        entry = &avi->index_entries[i];
        if (entry->stream_nr == stream_nr) {
               last_entry = entry;
                GST_DEBUG("No key found, use 1st entry size %d, offset %lld\n", last_entry->size, last_entry->offset);               
               break;
        }
     }
    } 
    return last_entry;
}
/*=============================================================================

FUNCTION:   mfw_avi_demux_get_keyframe   

DESCRIPTION: Get the next video key frame entry index for FFW/FBW.
            

ARGUMENTS PASSED:
        avi         -   The main structure of avi parser plugin context.
            
RETURN VALUE:
        None
      
PRE-CONDITIONS:
        None

POST-CONDITIONS:
		None

IMPORTANT NOTES:
        None

==============================================================================*/

static void *mfw_avi_demux_get_keyframe(MFW_GST_AVIDEMUX_INFO_T * avi)
{
    gst_avi_index_entry *entry = NULL;
    gint i;

    if (avi->trick_direction == FORWARD) {
        for (i = avi->current_entry; i < avi->index_size; i++) {
            entry = &avi->index_entries[i];

            if (entry->stream_nr == AVI_DEMUX_VIDEO_STREAM) {
                if ((entry->flags & AVIIF_KEYFRAME) == AVIIF_KEYFRAME) {
                    break;
                }
            }
        }
        avi->current_entry = i;
    }
    else if (avi->trick_direction == BACKWARD) {        
        for (i = avi->current_entry; i >= 0; i--) {
            entry = &avi->index_entries[i];

            if (entry->stream_nr == AVI_DEMUX_VIDEO_STREAM) {
                if ((entry->flags & AVIIF_KEYFRAME) == AVIIF_KEYFRAME) {
                    break;
                }
            }
        }
        avi->current_entry = i;
    }

}
/*=============================================================================

FUNCTION:   mfw_avi_demux_get_duration

DESCRIPTION:   Get duration from index table  

ARGUMENTS PASSED:
        avi         -   The main structure of avi parser plugin context.
        stream_nr   -   its 0 here , default for video .

RETURN VALUE:
        entry       -   pointer to entry.

PRE-CONDITIONS:
        None

POST-CONDITIONS:
                None

IMPORTANT NOTES:
        None

==============================================================================*/
static GstClockTime mfw_avi_demux_get_duration(MFW_GST_AVIDEMUX_INFO_T
                                                    * avi, gint stream_nr)
{
    gst_avi_index_entry *entry = NULL;
    gint i;
    GstClockTime duration = 0;
    guint64 ts=0, dur=0;

    /* Some information in avi header isn't acculate, so get from index table */
    for (i = avi->index_size-1; i >= 0;  i--) {
        entry = &avi->index_entries[i];
        if (entry->stream_nr == stream_nr) {
               break;  
        }
    }
    if(mfw_gst_avidemuxer_calc_timestamp(avi, entry, &ts, &dur))
            duration = ts + dur;
    return duration;
}
/*=============================================================================
FUNCTION:   	mfw_gst_avidemuxer_sink_event

DESCRIPTION:	Handles an event on the sink pad.

ARGUMENTS PASSED:
		pad        -    pointer to pad
		event      -    pointer to event
RETURN VALUE:
		TRUE       -	if event is sent to sink properly
		FALSE	   -	if event is not sent to sink properly

PRE-CONDITIONS:    None

POST-CONDITIONS:   None

IMPORTANT NOTES:   None
=============================================================================*/
static gboolean mfw_gst_avidemuxer_sink_event(GstPad * pad,
					      GstEvent * event)
{

    gboolean result = TRUE;
    GST_DEBUG("Into mfw_gst_avidemuxer_sink_event() function \n ");
    switch (GST_EVENT_TYPE(event)) {
    case GST_EVENT_NEWSEGMENT:
	{
	    GstFormat format;
	    gst_event_parse_new_segment(event, NULL, NULL, &format, NULL,
					NULL, NULL);
	    if (format == GST_FORMAT_TIME) {
		GST_DEBUG("\nCame to the FORMAT_TIME call\n");
		//gst_event_unref (event);
	    } else {
		GST_DEBUG("dropping newsegment event in format %s",
			  gst_format_get_name(format));
		gst_event_unref(event);
		result = TRUE;
	    }
	    break;
	}

    case GST_EVENT_EOS:
	{
	    GST_DEBUG("\nDemuxer: Sending EOS to Decoder\n");
	    result = gst_pad_push_event(pad, event);
	    if (result != TRUE) {
		GST_ERROR("\n Error in pushing the event, result is %d\n",
			  result);
	    }
	    break;
	}

    default:
	{
	    result = gst_pad_event_default(pad, event);
	    gst_event_unref(event);
	    break;
	}

    }

    GST_DEBUG("Out of mfw_gst_avidemuxer_sink_event() function \n ");
    return result;
}

/*=============================================================================
FUNCTION:   mfw_gst_avidemuxer_set_caps

DESCRIPTION:    this function handles the link with other plug-ins and used for
				capability negotiation  between pads

ARGUMENTS PASSED:
		pad        -    pointer to GstPad
		caps       -    pointer to GstCaps

RETURN VALUE:
		TRUE       -    if capabilities are set properly
		FALSE      -    if capabilities are not set properly
PRE-CONDITIONS:
		None

POST-CONDITIONS:
		None

IMPORTANT NOTES:
		None
=============================================================================*/
static gboolean mfw_gst_avidemuxer_set_caps(GstPad * pad, GstCaps * caps)
{

    MFW_GST_AVIDEMUX_INFO_T *demuxer = NULL;
    GstStructure *structure = gst_caps_get_structure(caps, 0);
    GST_DEBUG(" into mfw_avidemuxer_set_cap function \n");
    demuxer = MFW_GST_AVI_DEMUXER(GST_PAD_PARENT(pad));
    GST_DEBUG(" out of mfw_gst_avidemuxer_set_caps function \n");
    return TRUE;
}

/*=============================================================================
FUNCTION:               mfw_gst_avidemuxer_set_property

DESCRIPTION:            Sets the Property for the element.

ARGUMENTS PASSED:
						object  -> pointer on which property is set
						prop_id -> ID.
						value   -> Value to be set.
						pspec   -> Parameters to be set
RETURN VALUE:
						None

PRE-CONDITIONS:
						None

POST-CONDITIONS:
						None

IMPORTANT NOTES:
						None
=============================================================================*/
static void mfw_gst_avidemuxer_set_property(GObject * object,
					    guint prop_id,
					    const GValue * value,
					    GParamSpec * pspec)
{
    MFW_GST_AVIDEMUX_INFO_T * demuxer = MFW_GST_AVI_DEMUXER(object);
    unsigned char * drm_context = NULL;
    
    GST_DEBUG("\nInside the mfw_gst_avidemuxer_set_property function\n");
    
    switch (prop_id) {
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
            break;
    }
    return;   

    /* Not Yet Implemented. */

    GST_DEBUG
	("\nGoing out of the mfw_gst_avidemuxer_set_property function\n");
}

/*=============================================================================
FUNCTION:               mfw_gst_avidemuxer_get_property

DESCRIPTION:            Gets the Property of the element.

ARGUMENTS PASSED:
						object  -> pointer on which property is to be obtained.
						prop_id -> ID.
						value   -> Value to be get.
						pspec   -> Parameters to be get.

RETURN VALUE:
						None

PRE-CONDITIONS:
						None

POST-CONDITIONS:
						None

IMPORTANT NOTES:
						None
=============================================================================*/
static void mfw_gst_avidemuxer_get_property(GObject * object,
					    guint prop_id,
					    GValue * value,
					    GParamSpec * pspec)
{
    MFW_GST_AVIDEMUX_INFO_T * demuxer = MFW_GST_AVI_DEMUXER(object);
    GST_DEBUG("\nInside the mfw_gst_avidemuxer_get_property function\n");

    switch (prop_id) {
        default:
	        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
	    break;
    }

    GST_DEBUG
	("\nGoing out of the mfw_gst_avidemuxer_get_property function\n");
}

/*=============================================================================
FUNCTION:   		mfw_gst_avidemuxer_init

DESCRIPTION:    	create the pad template that has been registered with the
					element class in the _base_init and do library table
					initialization

ARGUMENTS PASSED:	demuxer - Main avi demuxer plugin structure.

RETURN VALUE:       None

PRE-CONDITIONS:     _base_init and _class_init are called

POST-CONDITIONS:    None

IMPORTANT NOTES:    None
=============================================================================*/
static void mfw_gst_avidemuxer_init(MFW_GST_AVIDEMUX_INFO_T * demuxer)
{

    GstPadTemplate *template = NULL;
    gboolean ret_val = TRUE;
    GstElementClass *klass = GST_ELEMENT_GET_CLASS(demuxer);
    GST_DEBUG("Entered into function mfw_gst_avidemuxer_init()\n");
    demuxer->sinkpad =
	gst_pad_new_from_template(gst_element_class_get_pad_template
				  (klass, "sink"), "sink");
    ret_val = gst_element_add_pad(GST_ELEMENT(demuxer), demuxer->sinkpad);
    if (ret_val == FALSE) {

	GST_DEBUG("Sink pad could not be added !!\n");

    }

    gst_pad_set_setcaps_function(demuxer->sinkpad,
				 mfw_gst_avidemuxer_set_caps);

    gst_pad_set_event_function(demuxer->sinkpad,
			       GST_DEBUG_FUNCPTR
			       (mfw_gst_avidemuxer_sink_event));

#define MFW_GST_AVI_DEMUXER_PLUGIN VERSION
    PRINT_PLUGIN_VERSION(MFW_GST_AVI_DEMUXER_PLUGIN);

    
    GST_DEBUG("Going out of function mfw_gst_avidemuxer_init()\n");
    return;
}

/*=============================================================================
FUNCTION:   	mfw_gst_avidemuxer_parser_init
	
DESCRIPTION:	
            This function initialize the parser .

ARGUMENTS PASSED:	
            demuxer 	- pointer to the avi demuxer element.
				
	 
RETURN VALUE:
		    GST_STATE_CHANGE_SUCCESS    -   success.
            GST_STATE_CHANGE_FAILURE    -   failure.

PRE-CONDITIONS:
		None

POST-CONDITIONS:
		None

IMPORTANT NOTES:
		None
=============================================================================*/
static GstStateChangeReturn
mfw_gst_avidemuxer_parser_init(MFW_GST_AVIDEMUX_INFO_T * demuxer)
{

    STD_BD_SET_T *stIn = NULL;
    STD_BD_SET_T *stOut = NULL;
    FSR_AVI_DEMUX_CTRL_T *stCtrl = NULL;
    gint i = 0;
    FSR_AVI_DEMUX_STAT_T *stStat = NULL;
    STD_RETURN_T uliResult = 0xFFFF;	//arbitrary value, not a valid return     

    /* Create and add the sink pad --
       Pad through which data comes into the demuxer element */

	/*** Initializing values ***/
    demuxer->init_parser = FALSE;
    demuxer->state = GST_STATE_NULL;
    demuxer->seek_event = NULL;
    //demuxer->isSeek = FALSE;
    demuxer->current_entry = 0;
	/***** Initializing the Demuxer Library. *****/

    stOut = demuxer->avi_str_content.stOut;
    stIn = &demuxer->avi_str_content.stIn;
    stCtrl = &demuxer->avi_str_content.stCtrl;
    stStat = &demuxer->avi_str_content.stStat;

    stOut[AVI_DEMUX_VIDEO_STREAM].pPtr =
	demuxer->avi_str_content.video_output;

    stOut[AVI_DEMUX_VIDEO_STREAM].usiCntBDSet = AVI_DEMUX_VIDEO_BUFFERS;

    stOut[AVI_DEMUX_AUDIO_STREAM].pPtr =
	demuxer->avi_str_content.audio_output;

    stOut[AVI_DEMUX_AUDIO_STREAM].usiCntBDSet = AVI_DEMUX_AUDIO_BUFFERS;

    /*
     *  Init stIn structure
     */
    stIn->pPtr = demuxer->avi_str_content.avi_input;
    stIn->usiCntBDSet = AVI_DEMUX_INPUT_BUFFERS;

    /*
     * Set the Control options in stCtrl
     */
    stCtrl->usiCntIn = 1;	/* one set in input space (avi data) */
    stCtrl->usiCntOut = 2;	/* two sets in output space (video and audio) */
    stCtrl->usiInRetMode = STD_MODE_NEED;
    /* return from process only if more input
       data is needed */
    stCtrl->usiOutRetMode = STD_MODE_NEED;
    /* (or) return from process only if more 
       output space is needed */

    stCtrl->uliRewFillIn = 0;	/* set initial rewind value to 0 */
    stCtrl->uliRewFillOut = 0;	/* set initial rewind value to 0 */
    stCtrl->usiRewBDIn = 0;	/* set initial rewind value to 0 */
    stCtrl->usiRewBDOut = 0;	/* set initial rewind value to 0 */

    stCtrl->paMemSelect = NULL;	/* no memory partitions will be used */
    stCtrl->usiCntMemSelect = 0;	/* no memory partitions will be used */

    stCtrl->MemPerInst.pPtr = NULL;	/* no channel data will be used */
    stCtrl->MemPerInst.uliLen = 0;	/* no channel data will be used */
    stCtrl->MemPerInst.uliFill = 0;	/* no channel data will be used */

    stStat->usiRewBDIn = 0;
    stStat->uliRewFillIn = 0;
    stStat->usiEndBitstream = 0;

     /**/ uliResult = FSR_AVI_DEMUX_Init(stCtrl);
    GST_DEBUG(" uliResult = %d\n", uliResult);
    if (uliResult != STD_RETURN_SUCCESS) {
	GST_ERROR("\nFATAL ERROR during AVI_DEMUX initialization\n");
	return GST_STATE_CHANGE_FAILURE;
    }
    return GST_STATE_CHANGE_SUCCESS;

}

static gboolean mfw_gst_avidemuxer_calc_timestamp(MFW_GST_AVIDEMUX_INFO_T * avi,
        gst_avi_index_entry * entry,
        guint64 * timestamp, guint64 * duration)
{
    avi_stream_context *stream = NULL;
    gint64 ts = 0;
    gint64 dur = 0;

    if (avi == NULL || entry == NULL)
        return FALSE;

    stream = &avi->avi_stream[entry->stream_nr];
    if (stream->strh.type == GST_RIFF_FCC_auds) {

        gboolean is_vbr = (stream->strh.samplesize == 0)&&(stream->strh.scale > 1);
        if (!is_vbr) {
            if (timestamp || duration)
                ts = gst_util_uint64_scale_int((guint64)entry->frames_before, 
                        GST_SECOND, stream->strf.auds.av_bps);
            if (duration)
                dur = gst_util_uint64_scale_int((guint64)entry->frames_before + entry->size, 
                        GST_SECOND, stream->strf.auds.av_bps);
        }
        //VBR
        else {
            if (timestamp || duration)
                ts = gst_util_uint64_scale_int((guint64)entry->frames_before * stream->strf.auds.blockalign, GST_SECOND, stream->strf.auds.rate);
            if (duration)
                dur = gst_util_uint64_scale_int((guint64)(entry->frames_before+1) * stream->strf.auds.blockalign, GST_SECOND, stream->strf.auds.rate);
        }
    } else {
        if (timestamp || duration)
            ts = gst_util_uint64_scale_int((guint64)entry->frames_before * 
                    stream->strh.scale, GST_SECOND, stream->strh.rate); 
        if (duration)
            dur = gst_util_uint64_scale_int((guint64)(entry->frames_before+1) * 
                    stream->strh.scale, GST_SECOND, stream->strh.rate); 
    }

    if (timestamp)
        *timestamp = ts;
    if (duration)
        *duration = dur - ts;
    return TRUE;
}

static guint64 avi_entry_calc_timestamp(MFW_GST_AVIDEMUX_INFO_T * avi,
        gst_avi_index_entry * entry)
{
    gint64 ts = 0;
    if (mfw_gst_avidemuxer_calc_timestamp(avi, entry, &ts, NULL)) {
        return ts;
    }
    return 0;
}

static guint64 avi_entry_calc_duration(MFW_GST_AVIDEMUX_INFO_T * avi,
        gst_avi_index_entry * entry)
{
    gint64 dur = 0;
    if (mfw_gst_avidemuxer_calc_timestamp(avi, entry, NULL, &dur)) {
        return dur;
    }
    return 0;
}

#ifdef DEMUX_ASYNC_MODE
static GstStateChangeReturn thread_func(gpointer data)
{
	GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;
	MFW_GST_AVIDEMUX_INFO_T *demuxer = MFW_GST_AVI_DEMUXER(data);
	GstPad *my_peer_pad = NULL;
	GstMessage *amessage = NULL;
	
	GST_DEBUG(" <mfw_gst_avidemuxer_change_state::thread_func> Thread entered! demuxer = %x\n",demuxer);
	
	//GST_OBJECT_LOCK (demuxer);
	/*:if src peerpad support pull mode, then active it for our initialization */
	if (gst_pad_check_pull_range(demuxer->sinkpad)) {
		my_peer_pad = gst_pad_get_peer(demuxer->sinkpad);
		gst_pad_activate_pull(my_peer_pad, TRUE);
	}
	
	
	ret = mfw_gst_avidemuxer_parse_header(demuxer);
	if (ret != GST_STATE_CHANGE_SUCCESS) {
		return ret;
		
	}
	
	gst_pad_activate_pull(my_peer_pad, FALSE);
	gst_object_unref(my_peer_pad);
	
	demuxer->isWork = TRUE;
	
	
    if (GST_ELEMENT_CLASS(parent_class)->change_state) {
		ret =
			GST_ELEMENT_CLASS(parent_class)->change_state(demuxer,
			GST_STATE_CHANGE_READY_TO_PAUSED);
    }
	
	GST_STATE (demuxer) = GST_STATE_PAUSED;
	GST_STATE_NEXT (demuxer) = GST_STATE_VOID_PENDING;
	GST_STATE_PENDING (demuxer) = GST_STATE_VOID_PENDING;
	GST_STATE_RETURN (demuxer) = GST_STATE_CHANGE_SUCCESS;
	
	amessage = gst_message_new_async_done (GST_OBJECT_CAST (demuxer));
	gst_element_post_message (GST_ELEMENT_CAST (demuxer), amessage);
	GST_DEBUG(" <mfw_gst_avidemuxer_change_state::thread_func> Thread exited!\n");
	return ret;
}
#endif
/*=============================================================================
FUNCTION:   	mfw_gst_avidemuxer_change_state
	
DESCRIPTION:	This function keeps track of different states of pipeline.

ARGUMENTS PASSED:	element 	- pointer to the avi demuxer element.
					transition	- state of the pipeline.
	 
RETURN VALUE:
		None

PRE-CONDITIONS:
		None

POST-CONDITIONS:
		None

IMPORTANT NOTES:
		None
=============================================================================*/
static GstStateChangeReturn mfw_gst_avidemuxer_change_state
    (GstElement * element, GstStateChange transition) {

    GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;
    MFW_GST_AVIDEMUX_INFO_T *demuxer = MFW_GST_AVI_DEMUXER(element);
    GstPad *my_peer_pad = NULL;

    GST_DEBUG("Entered into function mfw_gst_avidemuxer_change_state()\n");
	/***** UPWARDS *****/
    switch (transition) {
    case GST_STATE_CHANGE_NULL_TO_READY:
	/* Need to set all the variables to zero here. */
	{
	    GST_DEBUG
		("\nCame to Demuxer GST_STATE_CHANGE_NULL_TO_READY\n");

	}
	break;

    case GST_STATE_CHANGE_READY_TO_PAUSED:
	{

	    GST_DEBUG(" GST_STATE_CHANGE_READY_TO_PAUSED \n");

	    /* Registers a new tag type for the use with GStreamer's type system */
	    gst_tag_register(MFW_GST_TAG_WIDTH, GST_TAG_FLAG_DECODED,
			     G_TYPE_UINT, "image width",
			     "image width(pixel)", NULL);
	    gst_tag_register(MFW_GST_TAG_HEIGHT, GST_TAG_FLAG_DECODED,
			     G_TYPE_UINT, "image height",
			     "image height(pixel)", NULL);
	    gst_tag_register(MFW_GST_TAG_FRAMERATE, GST_TAG_FLAG_DECODED,
			     G_TYPE_FLOAT, "video framerate",
			     "number of video frames in a second", NULL);
	    gst_tag_register(MFW_GST_TAG_SAMPLING_FREQUENCY,
			     GST_TAG_FLAG_DECODED, G_TYPE_UINT,
			     "sampling frequency",
			     "number of audio samples per frame per second",
			     NULL);
	    gst_tag_register(MFW_GST_TAG_PRODUCT, GST_TAG_FLAG_DECODED,
			     G_TYPE_STRING, "product ", "product", NULL);
	    gst_tag_register(MFW_GST_TAG_SUBJECT, GST_TAG_FLAG_DECODED,
			     G_TYPE_STRING, "subject", "subject", NULL);

	    ret = mfw_gst_avidemuxer_parser_init(demuxer);
	    if (ret != GST_STATE_CHANGE_SUCCESS) {

		return ret;

	    }

            demuxer->isWork = FALSE;
	    demuxer->input_buf = NULL;
	    demuxer->input_buf_offset = 0;
	    demuxer->new_segment[0] = TRUE;
	    demuxer->new_segment[1] = TRUE;

	    demuxer->num_streams = 0;
	    demuxer->num_a_streams = 0;
	    demuxer->num_v_streams = 0;

	    demuxer->end_bitstream = 0;
	    demuxer->last_index_read = 0;
        demuxer->max_video_time = 0;
        demuxer->max_audio_time = 0;
        demuxer->video_eos = FALSE;
        demuxer->audio_eos = FALSE;
        demuxer->trick_mode = FALSE;
        demuxer->trick_direction = 0;
        demuxer->prvBufTime = GST_CLOCK_TIME_NONE;
	    demuxer->total_duration = 0;

        demuxer->videosent = FALSE;
	    GST_DEBUG
		("\nCame to Demuxer GST_STATE_CHANGE_READY_TO_PAUSED\n");

	    /* Allocate the memory for the input buffer. */
	    demuxer->input_buf =
		g_malloc(sizeof(char) * AVI_DEMUX_INPUT_BUFFER_SIZE);
	    if (demuxer->input_buf == NULL) {
		GST_ERROR
		    ("\nError in allocating memory for input buffer\n");

	    }

		/*************************************************************************/
	    /* Sets the given activate function for the pad. The activate function will
	       dispatch to activate_push or activate_pull to perform the actual activation.
	       Only makes sense to set on sink pads. */
		/*************************************************************************/
	    gst_pad_set_activate_function(demuxer->sinkpad,
					  mfw_gst_avidemuxer_activate);

		/*************************************************************************/
	    /* Sets the given activate_pull function for the pad. An activate_pull
	       function prepares the element and any upstream connections for pulling. */
		/*************************************************************************/
	    gst_pad_set_activatepull_function(demuxer->sinkpad,
					      GST_DEBUG_FUNCPTR
					      (mfw_gst_avidemuxer_activate_pull));


	    if (demuxer->init_parser == FALSE) {
#ifndef DEMUX_ASYNC_MODE
            /*:if src peerpad support pull mode, then active it for our initialization */
            if (gst_pad_check_pull_range(demuxer->sinkpad)) {
                my_peer_pad = gst_pad_get_peer(demuxer->sinkpad);
                gst_pad_activate_pull(my_peer_pad, TRUE);
            }


            ret = mfw_gst_avidemuxer_parse_header(demuxer);
            if (ret != GST_STATE_CHANGE_SUCCESS) {
                return ret;

            }

            gst_pad_activate_pull(my_peer_pad, FALSE);
            gst_object_unref(my_peer_pad);
#else
			gst_element_post_message (GST_ELEMENT_CAST (demuxer),
				gst_message_new_async_start (GST_OBJECT_CAST (demuxer), FALSE));
			
			// create a thread
			g_thread_create(thread_func, (gpointer)demuxer, FALSE, NULL);

			demuxer->current_entry = 0;

			return GST_STATE_CHANGE_ASYNC;		
#endif
	    }
	    demuxer->current_entry = 0;
	    demuxer->isWork = TRUE;
	}

	break;

    case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
	demuxer->state = GST_STATE_PLAYING;
	GST_DEBUG(" GST_STATE_CHANGE_PAUSED_TO_PLAYING \n");
	break;

    default:
	break;

    }
    if (GST_ELEMENT_CLASS(parent_class)->change_state) {
	ret =
	    GST_ELEMENT_CLASS(parent_class)->change_state(element,
							  transition);
	if (ret != GST_STATE_CHANGE_SUCCESS)
	    return ret;
    }

	 /***** DOWNWARDS *****/
    switch (transition) {

    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
	{

	    GST_DEBUG(" GST_STATE_CHANGE_PLAYING_TO_PAUSED \n");

	    break;
	}

    case GST_STATE_CHANGE_PAUSED_TO_READY:
	{
	    int i;

	    GST_DEBUG(" GST_STATE_CHANGE_PAUSED_TO_READY \n");

	    mfw_gst_avidemuxer_reset(demuxer);

	    FSR_AVI_DEMUX_Close(&demuxer->avi_str_content.stCtrl,
				&demuxer->avi_str_content.stStat);

	    if (NULL != demuxer->input_buf) {
		g_free(demuxer->input_buf);
		demuxer->input_buf = NULL;
	    }

	    for (i = 0; i < demuxer->num_streams; i++) {

		GstFlowReturn result = GST_FLOW_OK;
		avi_stream_context *stream_ptr = NULL;
		GstEvent *event = NULL;
		stream_ptr = &(demuxer->avi_stream[i]);

            if(stream_ptr->src_pad)
            {
                event = gst_event_new_eos();
        		result = gst_pad_push_event(stream_ptr->src_pad, event);
        		if (result != TRUE) {
        		    GST_ERROR
        			("\n Error in pushing the EOS event, result is %d\n",
        			 result);
        		}
        		gst_element_remove_pad(GST_ELEMENT(demuxer),
        				       stream_ptr->src_pad);
            }
	    }
	    break;
	}
    case GST_STATE_CHANGE_READY_TO_NULL:
	{
	    GST_DEBUG(" GST_STATE_CHANGE_READY_TO_NULL \n");
	}

	break;

    default:
	break;
    }
    GST_DEBUG("Out of function mfw_gst_avidemuxer_change_state()\n");
    return ret;
}

/*=============================================================================
FUNCTION:   		mfw_gst_avidemuxer_class_init

DESCRIPTION:		Initialise the class only once (specifying what arguments
					and virtual functions the class has and setting global
					state)

ARGUMENTS PASSED: 	klass   -	MFW_GST_AVIDEMUX_INFO_CLASS_T

RETURN VALUE:       None

PRE-CONDITIONS:     None

POST-CONDITIONS:    None

IMPORTANT NOTES:    None
=============================================================================*/
static void mfw_gst_avidemuxer_class_init(MFW_GST_AVIDEMUX_INFO_CLASS_T *
					  klass)
{

    GstElementClass *gstelement_class = GST_ELEMENT_CLASS(klass);
    GObjectClass *gobject_class = NULL;
    GParamSpec * pspec = NULL;

    GST_DEBUG("Entered into function mfw_gst_avidemuxer_class_init()\n");

    parent_class = g_type_class_peek_parent(klass);
    gstelement_class->change_state = mfw_gst_avidemuxer_change_state;

    GST_DEBUG("Out of function mfw_gst_avidemuxer_class_init()\n");
    return;
}

/*=============================================================================
FUNCTION:  			mfw_gst_avidemuxer_base_init

DESCRIPTION:		Elements details are registered with the plugin base_init,
					This function will initialize the class and child class 
					properties during each new child class creation.

ARGUMENTS PASSED:   Klass   -   void pointer

RETURN VALUE:       None

PRE-CONDITIONS:     None

POST-CONDITIONS:    None

IMPORTANT NOTES:    None
=============================================================================*/
static void mfw_gst_avidemuxer_base_init(MFW_GST_AVIDEMUX_INFO_CLASS_T *
					 klass)
{

    static GstElementDetails mfw_gst_avidemuxer_details =
	GST_ELEMENT_DETAILS("FSL Avi Demuxer",
			    "Codec/Demuxer",
			    "Demuxes an AVI file into Audio and Video files",
			    "Multimedia Team <mmsw@freescale.com>");

    GstElementClass *element_class = GST_ELEMENT_CLASS(klass);
    GstPadTemplate *videosrctempl = NULL;
    GstPadTemplate *audiosrctempl = NULL;
    GstCaps *audcaps = NULL;
    GstCaps *vidcaps = NULL;
    GST_DEBUG("Into mfw_gst_avidemuxer_base_init( ) function \n");
    /* Create audio source pad template */

    audcaps = gst_riff_create_audio_template_caps();
    gst_caps_append(audcaps,
		    gst_caps_new_simple("audio/x-avi-unknown", NULL));
    audiosrctempl =
	gst_pad_template_new("audio_%02d", GST_PAD_SRC, GST_PAD_SOMETIMES,
			     audcaps);

    vidcaps = gst_riff_create_video_template_caps();
    gst_caps_append(vidcaps,
		    gst_caps_new_simple("video/x-avi-unknown", NULL));

    
    videosrctempl =
	gst_pad_template_new("video_%02d", GST_PAD_SRC, GST_PAD_SOMETIMES,
			     vidcaps);

    gst_element_class_add_pad_template(element_class, audiosrctempl);
    gst_element_class_add_pad_template(element_class, videosrctempl);


    gst_element_class_add_pad_template(element_class,
				       gst_static_pad_template_get
				       (&mfw_gst_avidemuxer_sink_template_factory));

    gst_element_class_set_details(element_class,
				  &mfw_gst_avidemuxer_details);

    gst_object_unref(audiosrctempl);
    gst_object_unref(videosrctempl);
    GST_DEBUG("Out of mfw_gst_avidemuxer_base_init function \n");
    return;
}

/*=============================================================================
FUNCTION:			mfw_gst_type_avidemuxer_get_type

DESCRIPTION:    	intefaces are initiated in this function.you can register one
					or more interfaces  after having registered the type itself.

ARGUMENTS PASSED:   None

RETURN VALUE:		A numerical value, which represents the unique identifier of
					this element (avi demuxer)

PRE-CONDITIONS:     None

POST-CONDITIONS:    None

IMPORTANT NOTES:    None
=============================================================================*/
GType mfw_gst_type_avidemuxer_get_type(void)
{
    static GType avidemuxer_type = 0;
    if (!avidemuxer_type) {
	static const GTypeInfo avidemuxer_info = {
	    sizeof(MFW_GST_AVIDEMUX_INFO_CLASS_T),
	    mfw_gst_avidemuxer_base_init,
	    NULL,
	    (GClassInitFunc) mfw_gst_avidemuxer_class_init,
	    NULL,
	    NULL,
	    sizeof(MFW_GST_AVIDEMUX_INFO_T),
	    0,
	    (GInstanceInitFunc) mfw_gst_avidemuxer_init,
	};
	avidemuxer_type = g_type_register_static(GST_TYPE_ELEMENT,
						 "mfw_avidemuxer",
						 &avidemuxer_info, 0);
    }

    GST_DEBUG_CATEGORY_INIT(mfw_gst_avidemuxer_debug, "mfw_avidemuxer",
			    0, "Avi demuxer");
    return avidemuxer_type;
}

/*=============================================================================
FUNCTION:   plugin_init

DESCRIPTION:    special function , which is called as soon as the plugin or
				element is loaded and information returned by this function
				will be cached in central registry

ARGUMENTS PASSED:
		plugin     -    pointer to container that contains features loaded
						from shared object module

RETURN VALUE:
		return TRUE or FALSE depending on whether it loaded initialized any
		dependency correctly

PRE-CONDITIONS:		None

POST-CONDITIONS:    None

IMPORTANT NOTES:    None
=============================================================================*/
static gboolean plugin_init(GstPlugin * plugin)
{
    return gst_element_register(plugin, "mfw_avidemuxer",
				(GST_RANK_PRIMARY - 1),
				MFW_GST_TYPE_AVI_DEMUXER);
}

/*===========================================================================*/


/*****************************************************************************/
/*    This is used to define the entry point and meta data of plugin         */
/*****************************************************************************/
GST_PLUGIN_DEFINE(GST_VERSION_MAJOR,	/*   major version of Gstreamer   */
		  GST_VERSION_MINOR,	/*   minor version of Gstreamer   */
		  "mfw_avidemuxer",	/*   name of the plugin           */
		  "Demuxes the AVI streams ",	/* what plugin actually does */
		  plugin_init,	/*   first function to be called  */
		  VERSION,
		  GST_LICENSE_UNKNOWN,
		  "freescale semiconductor", "www.freescale.com ")
