/*
 * Copyright 2009 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of either or both of the following licenses:
 *
 * 1) the GNU Lesser General Public License version 3, as published by the
 * Free Software Foundation; and/or
 * 2) the GNU Lesser General Public License version 2.1, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the applicable version of the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of both the GNU Lesser General Public
 * License version 3 and version 2.1 along with this program.  If not, see
 * <http://www.gnu.org/licenses/>
 *
 * Authored by: Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "gadget-manager.h"

#include <gmodule.h>
#include <libgadget/gadget-source.h>

G_DEFINE_TYPE (GadgetManager, gadget_manager, G_TYPE_OBJECT);

#define GADGET_MANAGER_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  GADGET_TYPE_MANAGER, \
  GadgetManagerPrivate))

struct _GadgetManagerPrivate
{
  gchar *sources_dir;

  GSList *sources;
  GSList *modules;

  GSList *gadgets;
};

enum
{
  PROP_0,

  PROP_SOURCES_DIR
};

/* Globals */


/* Forwards */
static void          gadget_manager_set_sources_dir (GadgetManager *self,
    const gchar   *sources_dir);

static const gchar * gadget_manager_get_sources_dir (GadgetManager *self);


/* GObject stuff */
static void
gadget_manager_finalize (GObject *object)
{
  GadgetManagerPrivate *priv = GADGET_MANAGER_GET_PRIVATE (object);
  GSList *g;

  for (g = priv->gadgets; g; g = g->next)
    {
      if (GADGET_IS_PROXY (g->data))
        g_object_unref (g->data);
    }
  g_slist_free (priv->gadgets);
  priv->gadgets = NULL;

  if (priv->sources)
    {
      g_slist_foreach (priv->sources, (GFunc)g_object_unref, NULL);
      g_slist_free (priv->sources);
      priv->sources = NULL;
    }
  if (priv->modules)
    {
      g_slist_foreach (priv->modules, (GFunc)g_module_close, NULL);
      g_slist_free (priv->modules);
      priv->modules = NULL;
    }

  G_OBJECT_CLASS (gadget_manager_parent_class)->finalize (object);
}

static void
gadget_manager_set_property (GObject      *object,
                             guint         prop_id,
                             const GValue *value,
                             GParamSpec   *pspec)
{
  GadgetManager *self = GADGET_MANAGER (object);

  switch (prop_id)
    {
    case PROP_SOURCES_DIR:
      gadget_manager_set_sources_dir (self, g_value_get_string (value));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
gadget_manager_get_property (GObject    *object,
                             guint       prop_id,
                             GValue     *value,
                             GParamSpec *pspec)
{
  GadgetManager *self = GADGET_MANAGER (object);

  switch (prop_id)
    {
    case PROP_SOURCES_DIR:
      g_value_set_string (value,
                          gadget_manager_get_sources_dir (self));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
gadget_manager_class_init (GadgetManagerClass *klass)
{
  GObjectClass *obj_class = G_OBJECT_CLASS (klass);
  GParamSpec   *pspec;

  /* Overrides */
  obj_class->finalize     = gadget_manager_finalize;
  obj_class->set_property = gadget_manager_set_property;
  obj_class->get_property = gadget_manager_get_property;

  /* Add signals */
  pspec = g_param_spec_string ("sources-dir", "sources-dir",
                               "Directory that contains the sources modules",
                               SOURCES_DIR,
                               G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_SOURCES_DIR, pspec);

  /* Add Private struct */
  g_type_class_add_private (obj_class, sizeof (GadgetManagerPrivate));
}

static void
gadget_manager_init (GadgetManager *self)
{
  GadgetManagerPrivate *priv;

  priv = self->priv = GADGET_MANAGER_GET_PRIVATE (self);

  priv->sources = NULL;
  priv->modules = NULL;
}

/*
 * Private methods
 */
static void
gadget_manager_set_sources_dir (GadgetManager *self,
                                const gchar   *sources_dir)
{
  g_return_if_fail (GADGET_IS_MANAGER (self));
  g_return_if_fail (sources_dir);

  if (self->priv->sources_dir)
    g_free (self->priv->sources_dir);

  self->priv->sources_dir = g_strdup (sources_dir);
}

static const gchar *
gadget_manager_get_sources_dir (GadgetManager *self)
{
  g_return_val_if_fail (GADGET_IS_MANAGER (self), NULL);

  return self->priv->sources_dir;
}

/*
 * Public methods
 */
GadgetManager *
gadget_manager_get_default (void)
{
  static GadgetManager *self = NULL;

  if (G_UNLIKELY (!GADGET_IS_MANAGER (self)))
    {
      self = g_object_new (GADGET_TYPE_MANAGER, NULL);

      return self;
    }

  return g_object_ref (self);
}

void
gadget_manager_load_sources (GadgetManager *self)
{
  GadgetManagerPrivate *priv;
  GDir        *dir;
  const gchar *name;
  GError      *error = NULL;

  g_return_if_fail (GADGET_IS_MANAGER (self));
  priv = self->priv;

  if (priv->modules) /* Only init once in a session */
    return;

  if (G_UNLIKELY (priv->sources_dir == NULL))
    {
      g_critical ("Cannot load gadget sources: sources dir = NULL");
      return;
    }

  dir = g_dir_open (priv->sources_dir, 0, &error);
  if (error)
    {
      g_debug ("Cannot open sources directory: %s", error->message);
      g_error_free (error);
      return;
    }

  while ((name = g_dir_read_name (dir)) != NULL)
    {
      GModule *module;
      gchar   *fullpath;
      gadget_source_module_init_t init_func;

      if (!g_str_has_suffix (name, G_MODULE_SUFFIX))
        continue;

      fullpath = g_build_filename (priv->sources_dir, name, NULL);

      g_debug ("Loading Source: %s", fullpath);

      module = g_module_open (fullpath,
                              G_MODULE_BIND_LAZY | G_MODULE_BIND_LOCAL);
      if (!module)
        {
          g_warning ("Unable to load source %s: %s", name, g_module_error ());
          g_free (fullpath);
          continue;
        }

      if (g_module_symbol (module,
                           GADGET_SOURCE_INIT_SYMBOL,
                           (gpointer)&init_func))
        {
          GadgetSource *source = init_func ();

          if (!GADGET_IS_SOURCE (source))
            {
              g_warning ("Unable to initialize source: %s", name);
              g_module_close (module);
              g_free (fullpath);
              continue;
            }

          g_module_make_resident (module);

          priv->sources = g_slist_append (priv->sources, source);
          priv->modules = g_slist_append (priv->modules, module);

          g_free (fullpath);
        }
      else
        {
          g_warning ("Unable to find symbol %s in module %s",
                     GADGET_SOURCE_INIT_SYMBOL, name);
          g_free (fullpath);
          g_module_close (module);
        }
    }

  g_dir_close (dir);
}

gboolean
gadget_manager_can_handle_path (GadgetManager *self,
                                const gchar   *path)
{
  GadgetManagerPrivate *priv;
  gboolean  can_handle = FALSE;
  GSList   *s;

  g_return_val_if_fail (GADGET_IS_MANAGER (self), FALSE);
  g_return_val_if_fail (path, FALSE);
  priv = self->priv;

  for (s = priv->sources; s; s = s->next)
    {
      can_handle = gadget_source_can_handle_path (s->data, path);

      if (can_handle)
        break;
    }

  return can_handle;
}

GadgetProxy *
gadget_manager_load_gadget (GadgetManager *self,
                            const gchar   *path,
                            const gchar   *uid)
{
  GadgetManagerPrivate *priv;
  GadgetProxy *gadget = NULL;
  GSList      *s;

  g_return_val_if_fail (GADGET_IS_MANAGER (self), FALSE);
  g_return_val_if_fail (path, FALSE);
  priv = self->priv;

  for (s = priv->sources; s; s = s->next)
    {
      if (gadget_source_can_handle_path (s->data, path))
        {
          gadget = gadget_source_load_gadget (s->data, path, uid);
          break;
        }
    }

  if (gadget)
    priv->gadgets = g_slist_append (priv->gadgets, gadget);

  return gadget;
}

GList *
gadget_manager_get_available_gadgets (GadgetManager *self)
{
  GadgetManagerPrivate *priv;
  GList  *ret = NULL;
  GSList *s;

  g_return_val_if_fail (GADGET_IS_MANAGER (self), NULL);
  priv = self->priv;

  for (s = priv->sources; s; s = s->next)
    {
      GadgetSource *source = s->data;
      GList        *available = NULL;

      available = gadget_source_get_available_gadgets (source);

      ret = g_list_concat (ret, available);
    }

  return ret;
}

GadgetInfo *
gadget_info_new (void)
{
  return g_slice_new0 (GadgetInfo);
}

void
gadget_info_free (GadgetInfo *info)
{
  if (G_LIKELY (info))
    {
      g_free (info->name);
      g_free (info->description);
      g_free (info->path);
      g_free (info->author);
      g_free (info->version);
      if (info->icon)
        g_object_unref (info->icon);
      g_free (info->icon_name);

      g_slice_free (GadgetInfo, info);
    }
}
