/*
 * Copyright (C) 2009 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "ggadget-source.h"

#include <ggadget/gadget.h>
#include <ggadget/gadget_consts.h>
#include <ggadget/string_utils.h>
#include <ggadget/file_manager_interface.h>
#include <ggadget/file_manager_factory.h>
#include <ggadget/gtk/utilities.h>

#include <gio/gio.h>
#include <launcher/launcher.h>

#include "ggadget-gadget.h"
#include "libgadget-host.h"

using ggadget::LibgadgetHost;
using ggadget::StringMap;

using ggadget::kManifestName;
using ggadget::kManifestDescription;
using ggadget::kManifestAuthor;
using ggadget::kManifestId;
using ggadget::kManifestIcon;
using ggadget::kGadgetsIcon;

using ggadget::FileManagerInterface;
using ggadget::GetGlobalFileManager;

G_DEFINE_TYPE (GGadgetSource, ggadget_source, GADGET_TYPE_SOURCE);

#define GGADGET_SOURCE_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  GGADGET_TYPE_SOURCE, \
  GGadgetSourcePrivate))

struct _GGadgetSourcePrivate
{
  GList      *gadgets;
  GHashTable *icons_hash;
};

/* Forwards */
static gboolean      ggadget_source_can_handle_path (GadgetSource *source,
                                                     const gchar  *path);
static GadgetProxy * ggadget_source_load_gadget     (GadgetSource *source,
                                                     const gchar  *path,
                                                     const gchar  *uid);
static GList * ggadget_source_get_available_gadgets (GadgetSource *source);

/* Globals */

/* GObject stuff */
static void
ggadget_source_finalize (GObject *object)
{
  G_OBJECT_CLASS (ggadget_source_parent_class)->finalize (object);
}

static void
ggadget_source_class_init (GGadgetSourceClass *klass)
{
  GObjectClass      *obj_class = G_OBJECT_CLASS (klass);
  GadgetSourceClass *gad_class = GADGET_SOURCE_CLASS (klass);

  /* Overrides */
  obj_class->finalize        = ggadget_source_finalize;

  gad_class->can_handle_path       = ggadget_source_can_handle_path;
  gad_class->load_gadget           = ggadget_source_load_gadget;
  gad_class->get_available_gadgets = ggadget_source_get_available_gadgets;

  /* Install Properties */

  /* Add signals */

  /* Add Private struct */
  g_type_class_add_private (obj_class, sizeof (GGadgetSourcePrivate));
}

static void
ggadget_source_init (GGadgetSource *self)
{
  GGadgetSourcePrivate *priv;

  priv = self->priv = GGADGET_SOURCE_GET_PRIVATE (self);

  priv->gadgets = NULL;
}

GadgetSource *
ggadget_source_get_default (void)
{
#define ICON_LOC "/usr/share/icons/hicolor/128x128/apps/google-gadgets.png"
  static GadgetSource *source = NULL;

  if (G_UNLIKELY (!GGADGET_IS_SOURCE (source)))
    {
      void *const new_object = g_object_new (GGADGET_TYPE_SOURCE,
                                             "name", "GGadgets Source",
                                             "icon", ICON_LOC,
                                             NULL);
      source = static_cast <GadgetSource*> (new_object);
      return source;
    }

  return static_cast<GadgetSource *> (g_object_ref (source));
}

/*
 * Private methods
 */
static gboolean
ggadget_source_can_handle_path (GadgetSource *source,
                                const gchar  *path)
{
  /* Figure out if we can actually load this path */
  /* FIXME: We should also be able to load directories (unzipped .gg files) */
  if (g_str_has_suffix (path, ".gg")
      || g_str_has_suffix (path, ".gmanifest"))
    {
      return TRUE;
    }

  return FALSE;
}

static GadgetProxy *
ggadget_source_load_gadget (GadgetSource *source,
                            const gchar  *path,
                            const gchar  *uid)
{
  GGadgetSourcePrivate *priv = GGADGET_SOURCE (source)->priv;
  gchar *real_path;

  /* Check to see if the path exists */
  if (!g_file_test (path, G_FILE_TEST_EXISTS))
    {
      g_warning ("Unable to load gadget %s, path '%s' does not exist",
                 uid,
                 path);
      return NULL;
    }

  /* If gadget is not in /usr or ~/.config/ blah, move it there */
  if (!g_strstr_len (path, -1, DATADIR)
      && !g_strstr_len (path, -1, g_get_user_config_dir ()))
    {
      GFile *from, *to;
      gchar *gadget_dir;
      gchar *gadget_dest;
      LauncherFavorites *favs = launcher_favorites_get_default ();

      /* The gadget is being opened from outside of the launcher's known
       * space, so we'll move it to the launcher 
       */

      gadget_dir = g_build_filename (g_get_user_config_dir (),
                                     "netbook-launcher",
                                     "gadgets",
                                     NULL);
      g_mkdir_with_parents (gadget_dir, 0700);

      from = g_file_new_for_path (path);

      gadget_dest = g_build_filename (gadget_dir,
                                      g_file_get_basename (from),
                                      NULL);
      to = g_file_new_for_path (gadget_dest);
      
      if (g_file_copy (from, to,
                       G_FILE_COPY_OVERWRITE,
                       NULL,
                       NULL,
                       NULL,
                       NULL))
        {
          real_path = g_strdup (gadget_dest);

          launcher_favorites_set_string (favs, uid, "path", real_path);
        }
      else
        real_path = g_strdup (path);

      g_free (gadget_dest);
      g_free (gadget_dir);
      g_object_unref (to);
      g_object_unref (from);
    }
  else
    real_path = g_strdup (path);

  void * const new_gadget = g_object_new (G_TYPE_GADGET,
                                          "path", real_path,
                                          "uid", uid,
                                          NULL);
  priv->gadgets = g_list_append (priv->gadgets, new_gadget);

  g_free (real_path);

  return static_cast <GadgetProxy *> (new_gadget);
}

/*
 *
 *  Code to handle loading of existing gadgets
 *
 */
static void
element_start (GMarkupParseContext  *context,
               const gchar          *element_name,
               const gchar         **attribute_names,
               const gchar         **attribute_values,
               gpointer              user_data,
               GError              **error)
{
  gchar *uid = NULL;
  gchar *icon_name = NULL;
  gint   i = 0;

  if (g_strcmp0 (element_name, "plugin") != 0)
    return;

  for (i = 0; attribute_names[i]; i++)
    {
      if (g_strcmp0 (attribute_names[i], "name") == 0)
        uid = g_strdup (attribute_values[i]);
      else if (g_strcmp0 (attribute_names[i], "thumbnail_url") == 0)
        icon_name = g_strdup (attribute_values[i]);
    }

  if (uid && icon_name)
    {
      g_hash_table_replace (GGADGET_SOURCE (user_data)->priv->icons_hash,
                            uid,
                            icon_name);
    }
  else
    {
      g_free (uid);
      g_free (icon_name);
    }
}

static void
create_icon_names_hash (GGadgetSource *source,
                        gchar         *data,
                        gssize         size)
{
  static GMarkupParser  parser = { element_start, NULL, NULL, NULL, NULL };
  GMarkupParseContext  *ctx;

  ctx = g_markup_parse_context_new (&parser,
                                    G_MARKUP_PREFIX_ERROR_POSITION, 
                                    source, NULL);

  if (!g_markup_parse_context_parse (ctx, data, size, NULL))
    {
      g_error ("Unable to parse plugins.xml");
    }
  
  g_markup_parse_context_free (ctx);
}

static void
load_icon_names_hash (GGadgetSource *source)
{
  GGadgetSourcePrivate *priv = source->priv;
  GFile            *file;
  GFileInputStream *stream;
  GError           *error = NULL;
  GFileInfo        *info;
  
  if (priv->icons_hash)
    return;

  priv->icons_hash = g_hash_table_new_full (g_str_hash, g_str_equal,
                                            g_free, g_free);
  
  file = g_file_new_for_path (DATADIR"/libgadget-ggadgets/plugins.xml");
   
  stream = g_file_read (file, NULL, &error);
  if (error)
    {
      g_warning ("Unable to open plugins.xml for reading: %s",
                 error->message);
      g_error_free (error);
      g_object_unref (file);
      return;
    }

  info = g_file_query_info (file,
                            G_FILE_ATTRIBUTE_STANDARD_SIZE,
                            G_FILE_QUERY_INFO_NONE,
                            NULL, NULL);
  if (info != NULL)
    {
      goffset size;
      gssize  bytes_read;
      gchar  *data = NULL;

      size = g_file_info_get_size (info);
      data = (gchar *)g_malloc (size);

      bytes_read = g_input_stream_read (G_INPUT_STREAM (stream),
                                        data, size,
                                        NULL, NULL);

      if (bytes_read != -1)
        {
          create_icon_names_hash (source, data, size);  
        }

      g_free (data);
      g_object_unref (info);
    }

  g_object_unref (stream);
  g_object_unref (file);
}

static FileManagerInterface *
create_gadget_file_manager (const gchar *base_path)
{
  gchar *path;

  if (g_str_has_suffix (base_path, ".gmanifest"))
    path = g_path_get_dirname (base_path);
  else
    path = g_strdup (base_path);

  FileManagerInterface *fm = ggadget::CreateFileManager (path);
  g_free (path);

  return fm;
}

static GdkPixbuf *
get_pixbuf_for_icon_path (const gchar *icon_path, const gchar *gad_path)
{
  GdkPixbuf *pixbuf = NULL;
  std::string data;
  FileManagerInterface *gad_manager;

  gad_manager = create_gadget_file_manager (gad_path);
  if (gad_manager)
    {
      gad_manager->ReadFile(icon_path, &data);
      delete gad_manager;
    }

  if (data.empty())
    {
      FileManagerInterface *file_manager = GetGlobalFileManager ();
      if (file_manager)
        file_manager->ReadFile(kGadgetsIcon, &data);
    }

  if (!data.empty())
    {
      pixbuf = ggadget::gtk::LoadPixbufFromData(data);
    }

  if (pixbuf)
    {
      gint width, height;

      width = gdk_pixbuf_get_width (pixbuf);
      height = gdk_pixbuf_get_height (pixbuf);

      if (width < 96 || height < 96)
        {
          GdkPixbuf *temp = pixbuf;

          pixbuf = gdk_pixbuf_scale_simple (temp,
                                            96,
                                            ((gfloat)height/width) * 96.0,
                                            GDK_INTERP_HYPER);
          g_object_unref (temp);
        }

    }
  return pixbuf;
}

static void
get_icon_for_info (GGadgetSource *source,
                   GadgetInfo    *info,
                   const gchar   *manifest_icon_name,
                   const gchar   *path,
                   const gchar   *uid)
{
#define ICON_URL "http:/" "/desktop.google.com/plugins/images/%s"
  GGadgetSourcePrivate *priv = source->priv;
  gchar *icon_name;

  load_icon_names_hash (source);
  
  /* First check to see if the uid is available in the parsed list of
   * plugins.xml
   */
  icon_name = (gchar*)g_hash_table_lookup (priv->icons_hash, uid);
  if (icon_name)
    {
      if (g_str_has_prefix (icon_name, "http")
          || g_str_has_prefix (icon_name, "ftp"))
        info->icon_name = g_strdup (icon_name);
      else
        info->icon_name = g_strdup_printf (ICON_URL, icon_name);
    }
  else
    {
      info->icon = get_pixbuf_for_icon_path (manifest_icon_name, path);
    }
}

static GadgetInfo *
get_gadget_info_for_path (GGadgetSource *source, const gchar *path)
{
  GadgetInfo *info = NULL;
  StringMap manifest;

  if (!ggadget::Gadget::GetGadgetManifest (path, &manifest))
    return NULL;

  info = gadget_info_new ();
  info->name = g_strdup (manifest[kManifestName].c_str());
  info->description = g_strdup (manifest[kManifestDescription].c_str());
  info->path = g_strdup (path);
  info->author = g_strdup (manifest[kManifestAuthor].c_str());
  info->version = g_strdup (manifest[kManifestId].c_str());
  get_icon_for_info (source,
                     info,
                     manifest[kManifestIcon].c_str(),
                     path,
                     manifest[kManifestName].c_str());

  return info;
}

static void
load_gadgets_from_dir (GGadgetSource *source,
                       const gchar   *gadget_dir,
                       GHashTable    *gadget_table)
{
  GDir        *dir;
  const gchar *name;
  GError      *error = NULL;

  dir = g_dir_open (gadget_dir, 0, &error);
  if (error)
    {
      g_debug ("Skipping %s: %s", gadget_dir, error->message);
      g_error_free (error);
      return;
    }

  while ((name = g_dir_read_name (dir)) != NULL)
    {
      GadgetInfo *info;
      gchar *path;

      if (!g_str_has_suffix (name, ".gg")
          && !g_str_has_suffix (name, ".gmanifest"))
        continue;

      /* Never show these gadgets */
      if (g_strcmp0 (name, "igoogle.gg") == 0
          || g_strcmp0 (name, "designer.gg") == 0
          || g_strcmp0 (name, "google-gadget-browser.gg") == 0)
        continue;

      path = g_build_filename (gadget_dir, name, NULL);

      if ((info = get_gadget_info_for_path (source, path)))
        {
          GadgetInfo *last_info;

          last_info = (GadgetInfo *)g_hash_table_lookup (gadget_table, name);
          if (last_info)
            gadget_info_free (last_info);
          
          g_hash_table_replace (gadget_table, g_strdup (name), info);
        }

      g_free (path);
    }
  g_dir_close (dir);
}

static void
make_info_list (gchar *key, GadgetInfo *value, GList **list)
{
  *list = g_list_append (*list, value);
}

static GList *
ggadget_source_get_available_gadgets (GadgetSource *source)
{
  GGadgetSource        *self = GGADGET_SOURCE (source);
  GGadgetSourcePrivate *priv;
  GList                *ret = NULL;
  gchar                *local_dir;
  GHashTable           *gadget_table;

  g_return_val_if_fail (GGADGET_IS_SOURCE (source), NULL);
  priv = GGADGET_SOURCE (source)->priv;

  LibgadgetHost::Init ();

  gadget_table = g_hash_table_new_full (g_str_hash,
                                        g_str_equal,
                                        (GDestroyNotify)g_free,
                                        NULL);

  local_dir = g_build_filename (g_get_user_config_dir (),
                                "netbook-launcher",
                                "gadgets",
                                NULL);

  load_gadgets_from_dir (self, DATADIR"/gadgets", gadget_table);
  load_gadgets_from_dir (self, DATADIR"/google-gadgets", gadget_table);
  load_gadgets_from_dir (self, local_dir, gadget_table);

  /* Create a list out of the gadgets */
  g_hash_table_foreach (gadget_table, (GHFunc)make_info_list, &ret);
  
  g_hash_table_destroy (gadget_table);

  g_free (local_dir);

  return ret;
}

/*
 * Public methods
 */
