from __future__ import absolute_import

import re
import time
import locale

from contextlib import contextmanager
from datetime import datetime, timedelta


# Date time formats.  Assume no microseconds and no timezone.
ISO8601_FORMAT = '%Y-%m-%dT%H:%M:%S'
TWITTER_FORMAT = '%a %b %d %H:%M:%S %Y'


@contextmanager
def c_locale():
    locale.setlocale(locale.LC_TIME, 'C')
    try:
        yield
    finally:
        locale.setlocale(locale.LC_TIME, '')


def iso8601(t):
    return datetime.strptime(t, ISO8601_FORMAT)

def iso8601alt(t):
    return datetime.strptime(t, ISO8601_FORMAT.replace('T', ' '))

def twitter(t):
    return datetime.strptime(t, TWITTER_FORMAT)

def fromutctimestamp(t):
    return datetime.utcfromtimestamp(float(t))

def parsetime(t):
    """Parse an ISO 8601 datetime string and return seconds since epoch.

    This accepts either a naive (i.e. timezone-less) string or a timezone
    aware string.  The timezone must start with a + or - and must be followed
    by exactly four digits.  This string is parsed and converted to UTC.  This
    value is then converted to an integer seconds since epoch.
    """
    with c_locale():
        # In order to parse the UTC timezone (e.g. +0000), you'd think we
        # could just append %z on the format, but that doesn't work
        # across Python versions.  On some platforms in Python 2.7, %z is not
        # a valid format directive, and its use will raise a ValueError.
        #
        # In Python 3.2, strptime() is implemented in Python, so it *is*
        # supported everywhere, but we still can't rely on it because of the
        # non-ISO 8601 formats that some APIs use (I'm looking at you Twitter
        # and Facebook).  We'll use a regular expression to tear out the
        # timezone string and do the conversion ourselves.
        #
        # tz_offset is a list for two reasons.  First, so we can play games
        # with scoped access to the variable inside capture_tz() without the
        # use of `nonlocal` which doesn't exist in Python 2.  Also, we'll use
        # this as a cheap way of ensuring there aren't multiple matching
        # timezone strings in a single input string.
        tz_offset = []
        def capture_tz(match_object):
            tz_string = match_object.group('tz')
            if tz_string is not None:
                # It's possible that we'll see more than one substring
                # matching the timezone pattern.  It should be highly unlikely
                # so we won't test for that here, at least not now.
                #
                # The tz_offset is positive, so it must be subtracted from the
                # naive datetime in order to return it to UTC.  E.g.
                #   13:00 -0400 is 17:00 +0000
                # or
                # 1300 - (-0400 / 100)
                tz_offset.append(timedelta(hours=int(tz_string) / 100))
            # Return the empty string so as to remove the timezone pattern
            # from the string we're going to parse.
            return ''
        naive_t = re.sub(r'[ ]*(?P<tz>[-+]\d{4})', capture_tz, t)
        if len(tz_offset) == 0:
            tz_offset = timedelta()
        elif len(tz_offset) == 1:
            tz_offset = tz_offset[0]
        else:
            raise ValueError('Unsupported time string: {0}'.format(t))
        for parser in (iso8601, iso8601alt, twitter, fromutctimestamp):
            try:
                dt = parser(naive_t) - tz_offset
            except ValueError:
                pass
            else:
                break
        else:
            # Nothing matched.
            raise ValueError('Unsupported time string: {0}'.format(t))
        # We must have gotten a valid datetime.  Convert it to Epoch seconds.
        timetup = dt.timetuple()
        return int(time.mktime(timetup))
