from gwibber.microblog import network
from gwibber.microblog.util.auth import Authentication
from gwibber.microblog.util.time import parsetime

import logging
logger = logging.getLogger("Flickr")
logger.debug("Initializing.")

from gettext import lgettext as _

PROTOCOL_INFO = {
  "name": "Flickr",
  "version": 1.0,
  
  "config": [
    "username",
    "color",
    "receive_enabled",
  ],

  "authtype": "none",
  "color": "#C31A00",

  "features": [
    "receive",
    "images",
  ],

  "default_streams": [
    "images"
  ]
}

API_KEY = "36f660117e6555a9cbda4309cfaf72d0"
REST_SERVER = "http://api.flickr.com/services/rest"
BUDDY_ICON_URL = "http://farm%s.static.flickr.com/%s/buddyicons/%s.jpg"
IMAGE_URL = "http://farm%s.static.flickr.com/%s/%s_%s_%s.jpg"
IMAGE_PAGE_URL = "http://www.flickr.com/photos/%s/%s"

class Client:
  """Query Flickr, retrieve and convert data.
  
  The Client class is responsible for querying Flickr and 
  turning the data obtained into data that Gwibber can understand.
   
  """
  def __init__(self, acct):
    self.account = acct

  def _message(self, data):
    """Parses messages into Gwibber compatible forms.
    
    Arguments:
    data -- A data object obtained from Flickr containing a complete entry
    
    Returns: 
      m -- A data object compatible with inserting into the Gwibber database for that entry
      
    """
    m = {}
    m["mid"] = str(data["id"])
    m["service"] = "flickr"
    m["account"] = self.account["id"]
    m["time"] = parsetime(data["dateupload"])
    m["source"] = False
    m["text"] = data["title"]

    m["sender"] = {}
    m["sender"]["name"] = data["username"]
    m["sender"]["nick"] = data["ownername"]
    m["sender"]["is_me"] = data["ownername"] == self.account["username"]
    m["sender"]["id"] = data["owner"]
    m["sender"]["image"] = BUDDY_ICON_URL % (data["iconfarm"], data["iconserver"], data["owner"])
    m["sender"]["url"] = "http://www.flickr.com/people/%s" % (data["owner"])

    m["url"] = IMAGE_PAGE_URL % (data["owner"], data["id"])
    m["images"] = [{
      "full": IMAGE_URL % (data["farm"], data["server"], data["id"], data["secret"], "b"),
      "src": IMAGE_URL % (data["farm"], data["server"], data["id"], data["secret"], "m"),
      "thumb": IMAGE_URL % (data["farm"], data["server"], data["id"], data["secret"], "t"),
      "url": IMAGE_PAGE_URL % (data["owner"], data["id"])
    }]

    m["content"] = data["title"]
    m["html"] = data["title"]

    return m

  def _login(self):
    old_token = self.account.get("access_token", None)
    with self.account.login_lock:
      # Perform the login only if it wasn't already performed by another thread
      # while we were waiting to the lock
      if self.account.get("access_token", None) == old_token:
        self._locked_login(old_token)

    return "access_token" in self.account and \
        self.account["access_token"] != old_token

  def _locked_login(self, old_token):
    logger.debug("Re-authenticating" if old_token else "Logging in")

    auth = Authentication(self.account, logger)
    reply = auth.login()
    if reply and reply.has_key("AccessToken"):
      self.account["username"] = reply["username"]
      self.account["user_nsid"] = reply["user_nsid"]
      self.account["access_token"] = reply["AccessToken"]
      self.account["secret_token"] = reply["TokenSecret"]
    else:
      logger.error("Didn't find token in session: %s", (reply,))

  def _get(self, method, parse="message", post=False, **args):
    """Establishes a connection with Flickr and gets the data requested.
    
    Arguments:
    method -- The method to look up on Flickr
    parse -- The function to use to parse the data returned (message by default)
    post -- True is this is a send, False if a receive (False by default)
    **args -- Argument to be added to the URL when accessed
    
    Returns:
    A list of Gwibber compatible objects which have been parsed by the parse function.
      
    """
    args.update({"api_key": API_KEY, "method": method})
    data = network.Download(REST_SERVER, args, post).get_json()

    if parse == "message":
      return [self._message(m) for m in data["photos"]["photo"]]
    else: return data

  def _getNSID(self):
    """Gets user's own Flickr ID.
    
    Arguments:
    None
   
    Returns:
    The user's Flickr id.
    
    """
    if not "user_nsid" in self.account and not self._login():
      logstr = """%s: %s - %s""" % (PROTOCOL_INFO["name"], _("Authentication failed"), "Auth needs updating")
      logger.error("%s", logstr)
      return [{"error": {"type": "auth", "account": self.account, "message": _("Authentication failed, please re-authorize")}}]
    return self.account["user_nsid"]

  def __call__(self, opname, **args):
    return getattr(self, opname)(**args)

  def images(self):
    """Gets a stream of images from Flickr.
    
    Arguments:
    None
    
    Returns:
    A list of Gwibber compatible objects which have been 
    parsed by the parse function.
    
    """
    user_id=self._getNSID()
    if user_id:
      return self._get("flickr.photos.getContactsPublicPhotos",
      user_id=user_id, format="json", nojsoncallback="1",
      extras="date_upload,owner_name,icon_server")
    else: 
      logstr = """%s: %s""" % (PROTOCOL_INFO["name"], _("Failed to find account"))
      logger.error("%s", logstr)
      return [{"error": {"type": "auth", "account": self.account, "message": _("Failed to find account")}}]
