/*
 * Copyright (C) 2009 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 */
/**
 * SECTION:launcher-session
 * @short_description: Handles acquiring data from the current session
 * @include: launcher-session.h
 *
 * #LauncherSession objects exist to query data from the current session, 
 * Examples of this type of data include the workarea size and currently running applications
 */
 

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gdk/gdkx.h>
#include <X11/Xatom.h>
#include <X11/Xutil.h>

#include "launcher-application.h"
#include "launcher-menu.h"

#include "launcher-session.h"

G_DEFINE_TYPE (LauncherSession, launcher_session, G_TYPE_OBJECT)

#define LAUNCHER_SESSION_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE(obj, \
        LAUNCHER_TYPE_SESSION, LauncherSessionPrivate))

#define LAUNCHERAPP_ID "launcher-application-id"

struct _LauncherSessionPrivate
{
  /* Application variables */
  WnckScreen *screen;
  LauncherMenu   *menu;
  GSList *running_apps;

  /* Workarea variables */
  GdkWindow *root_window;
  gint left;
  gint top;
  gint right;
  gint bottom;
};

enum
{
  APP_STARTING,
  APP_OPENED,
  APP_CLOSED,

  WORKAREA_CHANGED,

  LAST_SIGNAL
};

static guint _session_signals[LAST_SIGNAL] = { 0 };

static LauncherSession *launcher_session = NULL;

/* Forwards */
static void on_application_opened (WnckScreen      *screen,
                                   WnckApplication *app,
                                   LauncherSession     *session);

static void on_application_closed (WnckScreen      *screen,
                                   WnckApplication *app,
                                   LauncherSession     *session);

static GdkFilterReturn launcher_session_property_filter (GdkXEvent *gdk_xevent,
    GdkEvent  *event,
    gpointer   data);

static void            update_net_workarea          (LauncherSession *session);

/* GObject Init */
static void
launcher_session_finalize (GObject *session)
{
  LauncherSessionPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_SESSION (session));
  priv = LAUNCHER_SESSION (session)->priv;

  if (priv->running_apps)
    {
      g_slist_foreach (priv->running_apps,
                       (GFunc)launcher_application_set_wnckapp, NULL);
      g_slist_foreach (priv->running_apps, (GFunc)launcher_application_unref, NULL);
      g_slist_free (priv->running_apps);
      priv->running_apps = NULL;
    }

  if (LAUNCHER_IS_MENU (priv->menu))
    {
      g_object_unref (priv->menu);
      priv->menu = NULL;
    }

  if (priv->root_window)
    {
      gdk_window_remove_filter (priv->root_window,
                                launcher_session_property_filter,
                                session);
      priv->root_window = NULL;
    }

  /* Note: We don't ref/unref priv->screen as-per-wnck-docs */
  priv->screen = NULL;

  G_OBJECT_CLASS (launcher_session_parent_class)->finalize (session);
}

static void
launcher_session_class_init (LauncherSessionClass *klass)
{
  GObjectClass *obj_class = G_OBJECT_CLASS (klass);

  obj_class->finalize = launcher_session_finalize;

  _session_signals[APP_STARTING] =
    g_signal_new ("application-starting",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (LauncherSessionClass, application_starting),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__POINTER,
                  G_TYPE_NONE, 1, G_TYPE_POINTER);

  _session_signals[APP_OPENED] =
    g_signal_new ("application-opened",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (LauncherSessionClass, application_opened),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__POINTER,
                  G_TYPE_NONE, 1, G_TYPE_POINTER);

  _session_signals[APP_CLOSED] =
    g_signal_new ("application-closed",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (LauncherSessionClass, application_closed),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__POINTER,
                  G_TYPE_NONE, 1, G_TYPE_POINTER);

  _session_signals[WORKAREA_CHANGED] =
    g_signal_new ("workarea-changed",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (LauncherSessionClass, workarea_changed),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  g_type_class_add_private (obj_class, sizeof (LauncherSessionPrivate));
}

static void
launcher_session_init (LauncherSession *session)
{
  LauncherSessionPrivate *priv;
  GdkWindow *root_window;

  priv = session->priv = LAUNCHER_SESSION_GET_PRIVATE (session);

  wnck_set_client_type (WNCK_CLIENT_TYPE_PAGER);

  /* Grab the menu, as that's what loads in the applications for us */
  priv->menu = launcher_menu_get_default ();

  /* Grab WnckScreen and connect to the important signals */
  priv->screen = wnck_screen_get_default ();
  g_signal_connect (priv->screen, "application-opened",
                    G_CALLBACK (on_application_opened), session);
  g_signal_connect (priv->screen, "application-closed",
                    G_CALLBACK (on_application_closed), session);

  /* Update the workarea */
  update_net_workarea (session);
  root_window = priv->root_window = gdk_get_default_root_window ();
  gdk_window_set_events (root_window, GDK_PROPERTY_CHANGE_MASK);
  gdk_window_add_filter (root_window, launcher_session_property_filter, session);
}

LauncherSession *
launcher_session_get_default (void)
{
  if (G_UNLIKELY (!LAUNCHER_IS_SESSION (launcher_session)))
    {
      launcher_session = g_object_new (LAUNCHER_TYPE_SESSION,
                                       NULL);
      return launcher_session;
    }

  return g_object_ref (launcher_session);
}

/*
 * Private methods
 */
/*
 * Taken from libwnck, LGPL 2.0
 */
static gchar *
latin1_to_utf8 (const gchar *latin1)
{
  GError *error = NULL;
  gsize read, written;
  gchar *utf8, *ret;

  utf8 = g_convert (latin1, -1, "UTF-8", "ISO-8859-1", &read, &written, &error);
  if (error)
    {
      g_debug ("errors converting latin1 to utf8: latin1=\"%s\", utf8=\"%s\", "
               "read=%zu, written=%zu, error=\"%s\".\n",
               latin1, utf8, read, written, error->message);
      g_error_free (error);
    }

  ret = g_utf8_strdown (utf8, -1);
  g_free (utf8);
  return ret;
}

static void
window_wmclass_get (gulong xid, gchar **res_name, gchar **class_name)
{
  XClassHint ch = {NULL, NULL};

  gdk_error_trap_push ();
  XGetClassHint (gdk_display, xid, &ch);
  gdk_error_trap_pop ();

  if (ch.res_name)
    {
      *res_name = latin1_to_utf8 (ch.res_name);
      XFree (ch.res_name);
    }
  else
    *res_name = NULL;

  if (ch.res_class)
    {
      *class_name = latin1_to_utf8 (ch.res_class);
      XFree (ch.res_class);
    }
  else
    *class_name = NULL;
}

static void
on_application_opened (WnckScreen      *screen,
                       WnckApplication *app,
                       LauncherSession     *session)
{
  LauncherSessionPrivate *priv;
  GSList             *apps, *a;
  gchar              *app_name = NULL;
  gchar              *res_name = NULL;
  gchar              *class_name = NULL;
  LauncherApplication    *nearmatch = NULL;
  LauncherApplication    *bestmatch = NULL;

  g_return_if_fail (LAUNCHER_IS_SESSION (session));
  g_return_if_fail (WNCK_IS_APPLICATION (app));
  priv = session->priv;

  /* Get the res_name and class_name to attempt a match */
  if (wnck_application_get_xid (app))
    window_wmclass_get (wnck_application_get_xid (app), &res_name, &class_name);

  /* Make sure we have at least something to match with */

  app_name = g_utf8_strdown (wnck_application_get_name (app), -1);

  /* Speeds things up a bit */
  if (g_strcmp0 (res_name, class_name) == 0)
    {
      g_free (class_name);
      class_name = NULL;
    }

  /* Try finding the application in the app list created by launcher-menu */
  apps = launcher_menu_get_applications (priv->menu);
  for (a = apps; a; a = a->next)
    {
      LauncherApplication *application = a->data;
      const gchar     *match_name;
      const gchar     *exec_string;

      match_name = launcher_application_get_match_name (application);
      exec_string = launcher_application_get_exec_string (application);

      /* Try straight app name match */
      if (app_name)
        {
          if (g_strstr_len (match_name, -1, app_name))
            {
              bestmatch = application;
              break;
            }
        }

      /* Check if the res_name is in the exec line of the application, 99% of
       * cases */
      if (res_name)
        {
          if (g_strstr_len (exec_string, -1, res_name)
              || g_strstr_len (match_name, -1, res_name))
            {
              nearmatch = application;
              continue;
            }
        }

      /* Check class_name next, this isn't as likely to match, but covers some
       * random cases (FIXME: Which ones?)
       */
      if (class_name)
        {
          if (g_strstr_len (exec_string, -1, class_name)
              || g_strstr_len (match_name, -1, class_name))
            {
              nearmatch = application;
              continue;
            }
        }
    }

  /* If we didn't match through the app name, then use the nearest match */
  if (!bestmatch)
    bestmatch = nearmatch;

  /* If we have a match, just ref it, otherwise create an app to represent the
   * opened application */
  if (bestmatch)
    {
      launcher_application_set_wnckapp (bestmatch, app);
      launcher_application_ref (bestmatch);

      g_print ("Application Matched: %s %s\n",
               launcher_application_get_name (bestmatch),
               wnck_application_get_name (app));
    }
  else
    {
      bestmatch = launcher_application_new_from_wnck_app (app);
      g_print ("Application Created: %s %s\n",
               launcher_application_get_name (bestmatch),
               wnck_application_get_name (app));
    }

  /* Add to the list of running apps, set the data property for the
   * WnckApplication class, so it's easy to locate when the application quits
   * and finally emit the signal notifying of the opened application
   */
  priv->running_apps = g_slist_append (priv->running_apps, bestmatch);
  g_object_set_data (G_OBJECT (app), LAUNCHERAPP_ID, bestmatch);
  g_signal_emit (session, _session_signals[APP_OPENED], 0, bestmatch);

  g_free (app_name);
  g_free (res_name);
  g_free (class_name);
}

static void
on_application_closed (WnckScreen      *screen,
                       WnckApplication *app,
                       LauncherSession     *session)
{
  LauncherSessionPrivate *priv;
  LauncherApplication *application;

  g_return_if_fail (LAUNCHER_IS_SESSION (session));
  g_return_if_fail (WNCK_IS_APPLICATION (app));
  priv = session->priv;

  application =(LauncherApplication*)g_object_get_data (G_OBJECT (app), LAUNCHERAPP_ID);

  if (application)
    {
      priv->running_apps = g_slist_remove (priv->running_apps, application);
      g_signal_emit (session, _session_signals[APP_CLOSED], 0, application);
      launcher_application_set_wnckapp (application, NULL);
      launcher_application_unref (application);
    }

  g_print ("Application Closed: %s\n", wnck_application_get_name (app));
}

/*
 * Workarea code is from Nautilus source: fm-session-icon-view.c
 * Copyright (C) 2000, 2001 Eazel, Inc.mou
 * License: LPGL v2 or later
 */
static GdkFilterReturn
launcher_session_property_filter (GdkXEvent *gdk_xevent,
                                  GdkEvent  *event,
                                  gpointer   data)
{
  XEvent *xevent = gdk_xevent;

  switch (xevent->type)
    {
    case PropertyNotify:
      if (xevent->xproperty.atom==gdk_x11_get_xatom_by_name ("_NET_WORKAREA"))
        {
          update_net_workarea (LAUNCHER_SESSION (data));
          g_signal_emit (data, _session_signals[WORKAREA_CHANGED], 0);
        }

      break;
    default:
      break;
    }
  return GDK_FILTER_CONTINUE;
}

static void
update_net_workarea (LauncherSession *session)
{
  LauncherSessionPrivate *priv;
  GdkScreen *screen;
  GdkWindow *window;
  GdkAtom    type_returned;
  int        format_returned;
  int        length_returned;
  long      *nworkareas = NULL;
  long      *workareas = NULL;

  g_return_if_fail (LAUNCHER_IS_SESSION (session));
  priv = session->priv;

  window = gdk_get_default_root_window ();
  screen = gdk_screen_get_default ();

  /* Find the number of sessions so we know how long the workareas array is
   * going to be (each session will have four elements in the workareas array
   * describing x, y, width, height)
   */
  gdk_error_trap_push ();
  if (!gdk_property_get (window,
                         gdk_atom_intern ("_NET_NUMBER_OF_DESKTOPS", FALSE),
                         gdk_x11_xatom_to_atom (XA_CARDINAL),
                         0, 4, FALSE,
                         &type_returned,
                         &format_returned,
                         &length_returned,
                         (guchar **)&nworkareas))
    {
      g_warning ("Cannot calculate _NET_NUMBER_OF_DESKTOPS");
    }
  if (gdk_error_trap_pop ()
      || nworkareas == NULL
      || type_returned != gdk_x11_xatom_to_atom (XA_CARDINAL)
      || format_returned != 32)
    {
      g_warning ("Cannot calculate _NET_NUMBER_OF_DESKTOPS");
    }

  /* Note : gdk_property_get() is broken (API documents admit
  * this). As a length argument, it expects the number of
  * _bytes_ of data you require. Internally, gdk_property_get
  * converts that value to a count of 32 bit (4 byte) elements.
  * However, the length returned is in bytes, but is calculated
  * via the count of returned elements * sizeof(long). This
  * means on a 64 bit system, the number of bytes you have to
  * request does not correspond to the number of bytes you get
  * back, and is the reason for the workaround below.
  */
  gdk_error_trap_push ();
  if (nworkareas == NULL
      || (*nworkareas < 1)
      || !gdk_property_get (window, gdk_atom_intern ("_NET_WORKAREA", FALSE),
                            gdk_x11_xatom_to_atom (XA_CARDINAL),
                            0, ((*nworkareas) * 4 * 4), FALSE,
                            &type_returned,
                            &format_returned,
                            &length_returned,
                            (guchar **)&workareas))
    {
      g_warning ("Cannot get _NET_WORKAREA");
    }

  if (gdk_error_trap_pop ()
      || workareas == NULL
      || type_returned != gdk_x11_xatom_to_atom (XA_CARDINAL)
      || ((*nworkareas) * 4 * sizeof (long)) != length_returned
      || format_returned != 32)
    {
      g_warning ("Cannot determine workarea, guessing panel layout");
      priv->left = 0;
      priv->top = 0;
      priv->right = gdk_screen_get_width (screen);
      priv->bottom = gdk_screen_get_height (screen);
    }
  else
    {
      gint left, right, top, bottom;
      gint screen_width, screen_height;
      gint i, n_items;

      left = right = top = bottom = 0;
      screen_width = gdk_screen_get_width (screen);
      screen_height = gdk_screen_get_height (screen);
      n_items = length_returned/sizeof (long);

      for (i = 0; i < n_items; i += 4)
        {
          int x      = workareas[i];
          int y      = workareas[i + 1];
          int width  = workareas[i + 2];
          int height = workareas[i + 3];

          if ((x + width) > screen_width || (y + height) > screen_height)
            continue;

          left   = MAX (left, x);
          right  = MAX (right, screen_width - width - x);
          top    = MAX (top, y);
          bottom = MAX (bottom, screen_height - height - y);
        }
      priv->left = left;
      priv->top = top;
      priv->right = right;
      priv->bottom = bottom;

      g_debug ("Workarea: (%d, %d) (%d, %d)", left, top, right, bottom);
    }

  if (nworkareas != NULL)
    g_free (nworkareas);
  if (workareas != NULL)
    g_free (workareas);
}

/*
 * Public Methods
 */

/**
 * launcher_session_get_running_applications:
 * @session: a #LauncherSession object
 *
 * This will produce a #GSList populated with currently running #LauncherApplication objects. 
 * This should be called after the mainloop has been started
 * <emphasis>This list should not be modified as it is owned by #LauncherSession</emphasis>
 *
 * Returns: A #GSList containing LauncherApplication objects
 */
 GSList *
launcher_session_get_running_applications (LauncherSession *session)
{
  g_return_val_if_fail (LAUNCHER_IS_SESSION (session), NULL);
  return session->priv->running_apps;
}

/**
 * launcher_session_get_workarea:
 * @session: A #LauncherSession session
 * @left: a gint reference
 * @top: a gint reference
 * @right: a gint reference
 * @bottom: a gint reference
 *
 * Gives you the current session _workarea_. This is the padding margin on the \
four sides of the screen caused by the panels
 */

void
launcher_session_get_workarea (LauncherSession *session,
                               gint        *left,
                               gint        *top,
                               gint        *right,
                               gint        *bottom)
{
  LauncherSessionPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_SESSION (session));
  priv = session->priv;

  if (left)
    *left = priv->left;
  if (top)
    *top = priv->top;
  if (right)
    *right = priv->right;
  if (bottom)
    *bottom = priv->bottom;
}
