/*
 * Copyright 2009 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of either or both of the following licenses:
 *
 * 1) the GNU Lesser General Public License version 3, as published by the
 * Free Software Foundation; and/or
 * 2) the GNU Lesser General Public License version 2.1, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the applicable version of the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of both the GNU Lesser General Public
 * License version 3 and version 2.1 along with this program.  If not, see
 * <http://www.gnu.org/licenses/>
 *
 * Authored by: Jay Taoko <jay.taoko@canonical.com>
 *
 */
/**
 * SECTION:ctk-perspective-correct-rendering
 * @short_description: Renders a polygon's projected vertices and apply the perspective correction to texture coordinates.
 *
 * Utility functions
 */

#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <math.h>
#include <string.h>

#include <GL/glew.h>
#include <GL/glxew.h>

#include <clutter/clutter.h>
#include "ctk-utils.h"
#include "ctk-arb-asm-private.h"
#include "ctk-perspective-correct-rendering.h"
#include <gio/gio.h>

/**
 * ctk_render_projected_polygon
 *
 * @V: An array of CtkProjectedTextureVertex
 * @tex_id: OpenGL texture id
 * @window_w: width of the stage (width of window)
 * @window_h: height of the stage (height of window)
 **/
void ctk_render_projected_polygon (CtkProjectedTextureVertex *V, guint tex_id, gint window_w, gint window_h)
{
  gfloat VtxBuffer[] = {
     // X    Y       Z        W       S              T              R              Q              R                      G                        B                       A
     V[0].x, V[0].y, V[0].z, 1.0f,  V[0].s/V[0].w, V[0].t/V[0].w, V[0].r/V[0].w, V[0].q/V[0].w, V[0].color.red/255.0f, V[0].color.green/255.0f, V[0].color.blue/255.0f, V[0].color.alpha/255.0f,
     V[1].x, V[1].y, V[1].z, 1.0f,  V[1].s/V[1].w, V[1].t/V[1].w, V[1].r/V[1].w, V[1].q/V[1].w, V[1].color.red/255.0f, V[1].color.green/255.0f, V[1].color.blue/255.0f, V[1].color.alpha/255.0f,
     V[2].x, V[2].y, V[2].z, 1.0f,  V[2].s/V[2].w, V[2].t/V[2].w, V[2].r/V[2].w, V[2].q/V[2].w, V[2].color.red/255.0f, V[2].color.green/255.0f, V[2].color.blue/255.0f, V[2].color.alpha/255.0f,
     V[3].x, V[3].y, V[3].z, 1.0f,  V[3].s/V[3].w, V[3].t/V[3].w, V[3].r/V[3].w, V[3].q/V[3].w, V[3].color.red/255.0f, V[3].color.green/255.0f, V[3].color.blue/255.0f, V[3].color.alpha/255.0f,
  };

//   printf("V0: %f, %f, %f, %f \n", V[0].x, V[0].y, V[0].z, V[0].w);
//   printf("V1: %f, %f, %f, %f \n", V[1].x, V[1].y, V[1].z, V[1].w);
//   printf("V2: %f, %f, %f, %f \n", V[2].x, V[2].y, V[2].z, V[2].w);
//   printf("V3: %f, %f, %f, %f \n", V[3].x, V[3].y, V[3].z, V[3].w);
//   gfloat VtxBuffer[] = {
//      // X    Y     Z     W     U0    V0                R     G     B     A
//      5,      26,    0.0f, 1.0f, 0.0f, 1.0f, 0.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f,
//      5,      76,  0.0f, 1.0f, 0.0f, 0.0f, 0.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f,
//      55,    76,  0.0f, 1.0f, 1.0f, 0.0f, 0.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f,
//      55,    26,    0.0f, 1.0f, 1.0f, 1.0f, 0.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f,
//   };

  gint VertexLocation        = ATTRIB_POSITION;
  gint TextureCoord0Location = ATTRIB_COLOR_TEXCOORD0;
  gint VertexColorLocation   = ATTRIB_COLOR;

  CHECKGL(glBindBufferARB(GL_ARRAY_BUFFER_ARB, 0));
  CHECKGL(glBindBufferARB(GL_ELEMENT_ARRAY_BUFFER_ARB, 0));

  // Use the shader
  {
    CHECKGL( glEnable(GL_VERTEX_PROGRAM_ARB) );
    CHECKGL( glBindProgramARB(GL_VERTEX_PROGRAM_ARB, g_shTextureUVPerspDivision_asm->shvert) );
    CHECKGL( glEnable(GL_FRAGMENT_PROGRAM_ARB) );
    CHECKGL( glBindProgramARB(GL_FRAGMENT_PROGRAM_ARB, g_shTextureUVPerspDivision_asm->shfrag) );
  }

  CHECKGL( glEnableVertexAttribArrayARB (VertexLocation) );
  CHECKGL( glVertexAttribPointerARB ((GLuint)VertexLocation, 4, GL_FLOAT, GL_FALSE, 48, VtxBuffer) );

  if(TextureCoord0Location != -1)
  {
    CHECKGL( glEnableVertexAttribArrayARB (TextureCoord0Location) );
    CHECKGL( glVertexAttribPointerARB ((GLuint)TextureCoord0Location, 4, GL_FLOAT, GL_FALSE, 48, VtxBuffer + 4) );
  }

  if(VertexColorLocation != -1)
  {
    CHECKGL( glEnableVertexAttribArrayARB (VertexColorLocation) );
    CHECKGL( glVertexAttribPointerARB ((GLuint)VertexColorLocation, 4, GL_FLOAT, GL_FALSE, 48, VtxBuffer + 8) );
  }

  CHECKGL( glActiveTextureARB(GL_TEXTURE0) );
  CHECKGL( glEnable (GL_TEXTURE_2D) );
  CHECKGL( glBindTexture(GL_TEXTURE_2D, tex_id) );
  
  // Preserve model-view and projection matrices
  {
    CHECKGL( glMatrixMode(GL_PROJECTION) );
    CHECKGL( glPushMatrix() );
    CHECKGL( glLoadIdentity() );
    CHECKGL( glOrtho(0, window_w, window_h, 0, -1, 1) );

    CHECKGL( glMatrixMode(GL_MODELVIEW) );
    CHECKGL( glPushMatrix() );
    CHECKGL( glLoadIdentity() );
  }

  gint blend;
  gint blend_src;
  gint blend_dst;
  glGetIntegerv(GL_BLEND, &blend);
  glGetIntegerv(GL_BLEND_SRC, &blend_src);
  glGetIntegerv(GL_BLEND_DST, &blend_dst);

  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  //CHECKGL( glDrawArrays(GL_QUADS, 0, 4) );
  glBegin(GL_QUADS);
  glColor4f(V[0].color.red/255.0f, V[0].color.green/255.0f, V[0].color.blue/255.0f, V[0].color.alpha/255.0f);
  glMultiTexCoord4f(GL_TEXTURE0, V[0].s/V[0].w, V[0].t/V[0].w, V[0].r/V[0].w, V[0].q/V[0].w);
  glVertex4f(V[0].x, V[0].y, V[0].z, 1.0f);
  
  glColor4f(V[1].color.red/255.0f, V[1].color.green/255.0f, V[1].color.blue/255.0f, V[1].color.alpha/255.0f);
  glMultiTexCoord4f(GL_TEXTURE0, V[1].s/V[1].w, V[1].t/V[1].w, V[1].r/V[1].w, V[1].q/V[1].w);
  glVertex4f(V[1].x, V[1].y, V[1].z, 1.0f);
  
  glColor4f(V[2].color.red/255.0f, V[2].color.green/255.0f, V[2].color.blue/255.0f, V[2].color.alpha/255.0f);
  glMultiTexCoord4f(GL_TEXTURE0, V[2].s/V[2].w, V[2].t/V[2].w, V[2].r/V[2].w, V[2].q/V[2].w);
  glVertex4f(V[2].x, V[2].y, V[2].z, 1.0f);
  
  glColor4f(V[3].color.red/255.0f, V[3].color.green/255.0f, V[3].color.blue/255.0f, V[3].color.alpha/255.0f);
  glMultiTexCoord4f(GL_TEXTURE0, V[3].s/V[3].w, V[3].t/V[3].w, V[3].r/V[3].w, V[3].q/V[3].w);
  glVertex4f(V[3].x, V[3].y, V[3].z, 1.0f);
  glEnd();  

  // Restore model-view and projection matrices
  {
    CHECKGL( glMatrixMode(GL_PROJECTION) );
    CHECKGL( glPopMatrix() );
    CHECKGL( glMatrixMode(GL_MODELVIEW) );
    CHECKGL( glPopMatrix() );
  }

  glBlendFunc(blend_src, blend_dst);

  CHECKGL( glDisableVertexAttribArrayARB(VertexLocation) );
  if(TextureCoord0Location != -1)
    CHECKGL( glDisableVertexAttribArrayARB(TextureCoord0Location) );
  if(VertexColorLocation != -1)
    CHECKGL( glDisableVertexAttribArrayARB(VertexColorLocation) );


  // Stop using the shader program
  CHECKGL( glDisable(GL_VERTEX_PROGRAM_ARB) );
  CHECKGL( glDisable(GL_FRAGMENT_PROGRAM_ARB) );
}

/**
 * ctk_render_projected_polygon
 *
 * @V: An array of CtkProjectedTextureVertex
 * @tex_mask_id: OpenGL texture id
 * @window_w: width of the stage (width of window)
 * @window_h: height of the stage (height of window)
 **/
void ctk_render_projected_polygon_2Tex (CtkProjectedTextureVertex *V, guint tex_mask_id, guint tex_id, gint window_w, gint window_h)
{
  gfloat VtxBuffer[] = {
     // X    Y       Z        W       S              T              R              Q              R                      G                        B                       A
     V[0].x, V[0].y, V[0].z, 1.0f,  V[0].s/V[0].w, V[0].t/V[0].w, V[0].r/V[0].w, V[0].q/V[0].w, V[0].color.red/255.0f, V[0].color.green/255.0f, V[0].color.blue/255.0f, V[0].color.alpha/255.0f,
     V[1].x, V[1].y, V[1].z, 1.0f,  V[1].s/V[1].w, V[1].t/V[1].w, V[1].r/V[1].w, V[1].q/V[1].w, V[1].color.red/255.0f, V[1].color.green/255.0f, V[1].color.blue/255.0f, V[1].color.alpha/255.0f,
     V[2].x, V[2].y, V[2].z, 1.0f,  V[2].s/V[2].w, V[2].t/V[2].w, V[2].r/V[2].w, V[2].q/V[2].w, V[2].color.red/255.0f, V[2].color.green/255.0f, V[2].color.blue/255.0f, V[2].color.alpha/255.0f,
     V[3].x, V[3].y, V[3].z, 1.0f,  V[3].s/V[3].w, V[3].t/V[3].w, V[3].r/V[3].w, V[3].q/V[3].w, V[3].color.red/255.0f, V[3].color.green/255.0f, V[3].color.blue/255.0f, V[3].color.alpha/255.0f,
  };

//   printf("V0: %f, %f, %f, %f \n", V[0].x, V[0].y, V[0].z, V[0].w);
//   printf("V1: %f, %f, %f, %f \n", V[1].x, V[1].y, V[1].z, V[1].w);
//   printf("V2: %f, %f, %f, %f \n", V[2].x, V[2].y, V[2].z, V[2].w);
//   printf("V3: %f, %f, %f, %f \n", V[3].x, V[3].y, V[3].z, V[3].w);
//   gfloat VtxBuffer[] = {
//      // X    Y     Z     W     U0    V0                R     G     B     A
//      5,      26,    0.0f, 1.0f, 0.0f, 1.0f, 0.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f,
//      5,      76,  0.0f, 1.0f, 0.0f, 0.0f, 0.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f,
//      55,    76,  0.0f, 1.0f, 1.0f, 0.0f, 0.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f,
//      55,    26,    0.0f, 1.0f, 1.0f, 1.0f, 0.0f, 1.0f, 1.0f, 1.0f, 1.0f, 1.0f,
//   };

  gint VertexLocation        = ATTRIB_POSITION;
  gint TextureCoord0Location = ATTRIB_COLOR_TEXCOORD0;
  gint VertexColorLocation   = ATTRIB_COLOR;

  CHECKGL(glBindBufferARB(GL_ARRAY_BUFFER_ARB, 0));
  CHECKGL(glBindBufferARB(GL_ELEMENT_ARRAY_BUFFER_ARB, 0));

  // Use the shader
  {
    CHECKGL( glEnable(GL_VERTEX_PROGRAM_ARB) );
    CHECKGL( glBindProgramARB(GL_VERTEX_PROGRAM_ARB, g_shTextureUVPerspDivision_2Tex_asm->shvert) );
    CHECKGL( glEnable(GL_FRAGMENT_PROGRAM_ARB) );
    CHECKGL( glBindProgramARB(GL_FRAGMENT_PROGRAM_ARB, g_shTextureUVPerspDivision_2Tex_asm->shfrag) );
  }

  CHECKGL( glEnableVertexAttribArrayARB (VertexLocation) );
  CHECKGL( glVertexAttribPointerARB ((GLuint)VertexLocation, 4, GL_FLOAT, GL_FALSE, 48, VtxBuffer) );

  if(TextureCoord0Location != -1)
  {
    CHECKGL( glEnableVertexAttribArrayARB (TextureCoord0Location) );
    CHECKGL( glVertexAttribPointerARB ((GLuint)TextureCoord0Location, 4, GL_FLOAT, GL_FALSE, 48, VtxBuffer + 4) );
  }

  if(VertexColorLocation != -1)
  {
    CHECKGL( glEnableVertexAttribArrayARB (VertexColorLocation) );
    CHECKGL( glVertexAttribPointerARB ((GLuint)VertexColorLocation, 4, GL_FLOAT, GL_FALSE, 48, VtxBuffer + 8) );
  }

  CHECKGL( glActiveTextureARB(GL_TEXTURE0) );
  CHECKGL( glEnable (GL_TEXTURE_2D) );
  CHECKGL( glBindTexture(GL_TEXTURE_2D, tex_mask_id) );
  
  CHECKGL( glActiveTextureARB(GL_TEXTURE1) );
  CHECKGL( glEnable (GL_TEXTURE_2D) );
  CHECKGL( glBindTexture(GL_TEXTURE_2D, tex_id) );
  
  // Preserve model-view and projection matrices
  {
    CHECKGL( glMatrixMode(GL_PROJECTION) );
    CHECKGL( glPushMatrix() );
    CHECKGL( glLoadIdentity() );
    CHECKGL( glOrtho(0, window_w, window_h, 0, -1, 1) );

    CHECKGL( glMatrixMode(GL_MODELVIEW) );
    CHECKGL( glPushMatrix() );
    CHECKGL( glLoadIdentity() );
  }

  gint blend;
  gint blend_src;
  gint blend_dst;
  glGetIntegerv(GL_BLEND, &blend);
  glGetIntegerv(GL_BLEND_SRC, &blend_src);
  glGetIntegerv(GL_BLEND_DST, &blend_dst);

  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  //CHECKGL( glDrawArrays(GL_QUADS, 0, 4) );
  glBegin(GL_QUADS);
  glColor4f(V[0].color.red/255.0f, V[0].color.green/255.0f, V[0].color.blue/255.0f, V[0].color.alpha/255.0f);
  glMultiTexCoord4f(GL_TEXTURE0, V[0].s/V[0].w, V[0].t/V[0].w, V[0].r/V[0].w, V[0].q/V[0].w);
  glVertex4f(V[0].x, V[0].y, V[0].z, 1.0f);
  
  glColor4f(V[1].color.red/255.0f, V[1].color.green/255.0f, V[1].color.blue/255.0f, V[1].color.alpha/255.0f);
  glMultiTexCoord4f(GL_TEXTURE0, V[1].s/V[1].w, V[1].t/V[1].w, V[1].r/V[1].w, V[1].q/V[1].w);
  glVertex4f(V[1].x, V[1].y, V[1].z, 1.0f);
  
  glColor4f(V[2].color.red/255.0f, V[2].color.green/255.0f, V[2].color.blue/255.0f, V[2].color.alpha/255.0f);
  glMultiTexCoord4f(GL_TEXTURE0, V[2].s/V[2].w, V[2].t/V[2].w, V[2].r/V[2].w, V[2].q/V[2].w);
  glVertex4f(V[2].x, V[2].y, V[2].z, 1.0f);
  
  glColor4f(V[3].color.red/255.0f, V[3].color.green/255.0f, V[3].color.blue/255.0f, V[3].color.alpha/255.0f);
  glMultiTexCoord4f(GL_TEXTURE0, V[3].s/V[3].w, V[3].t/V[3].w, V[3].r/V[3].w, V[3].q/V[3].w);
  glVertex4f(V[3].x, V[3].y, V[3].z, 1.0f);
  glEnd();

  // Restore model-view and projection matrices
  {
    CHECKGL( glMatrixMode(GL_PROJECTION) );
    CHECKGL( glPopMatrix() );
    CHECKGL( glMatrixMode(GL_MODELVIEW) );
    CHECKGL( glPopMatrix() );
  }

  glBlendFunc(blend_src, blend_dst);

  CHECKGL( glDisableVertexAttribArrayARB(VertexLocation) );
  if(TextureCoord0Location != -1)
    CHECKGL( glDisableVertexAttribArrayARB(TextureCoord0Location) );
  if(VertexColorLocation != -1)
    CHECKGL( glDisableVertexAttribArrayARB(VertexColorLocation) );

  CHECKGL( glActiveTextureARB(GL_TEXTURE1) );
  CHECKGL( glDisable (GL_TEXTURE_2D) );

  // Stop using the shader program
  CHECKGL( glDisable(GL_VERTEX_PROGRAM_ARB) );
  CHECKGL( glDisable(GL_FRAGMENT_PROGRAM_ARB) );
}

void ctk_get_actor_screen_coord (CtkActor* actor, gfloat *x, gfloat *y, gfloat *width, gfloat *height)
{
  gfloat actor_screen_width, actor_screen_height;
  gfloat actor_screen_x, actor_screen_y;

  ClutterVertex vtx[4];
  ctk_get_actor_screen_position (CTK_ACTOR(actor),
    &actor_screen_x,
    &actor_screen_y,
    &actor_screen_width,
    &actor_screen_height, vtx);
      
  *x = actor_screen_x;
  *y = actor_screen_y;
  *width = actor_screen_width;
  *height = actor_screen_height;
}

/**
 * ctk_create_opengl_texture_from_pixbuf
 *
 * GdkPixbuf: a GdkPixbuf of the image
 *
 **/

guint ctk_create_opengl_texture_from_pixbuf (GdkPixbuf *pixbuf)
{
  g_return_val_if_fail (GDK_IS_PIXBUF (pixbuf), 0);

  guint opengl_id = 0;
  int unpack = 0;
  unsigned char *pixelbuffer = 0;
  pixelbuffer = gdk_pixbuf_get_pixels(pixbuf);

  int texture_width = gdk_pixbuf_get_width(pixbuf);
  int texture_height = gdk_pixbuf_get_height(pixbuf);

  gint nChannel = gdk_pixbuf_get_n_channels(pixbuf);

  CHECKGL (glGetIntegerv(GL_UNPACK_ALIGNMENT, &unpack));
  CHECKGL (glPixelStorei(GL_UNPACK_ALIGNMENT, 4));
  CHECKGL (glPixelStorei(GL_UNPACK_ROW_LENGTH, 0));
  CHECKGL (glPixelStorei(GL_UNPACK_IMAGE_HEIGHT, 0));
  CHECKGL (glPixelStorei(GL_UNPACK_SKIP_PIXELS, 0));
  CHECKGL (glPixelStorei(GL_UNPACK_SKIP_ROWS, 0));

  CHECKGL (glActiveTextureARB (GL_TEXTURE0));

  CHECKGL (glGenTextures (1, &opengl_id));

  CHECKGL (glBindTexture (GL_TEXTURE_2D, opengl_id));

  if (nChannel == 3)
    {
      CHECKGL (glTexImage2D (GL_TEXTURE_2D,
                         0,
                         GL_RGB,
                         (gint) texture_width,
                         (gint) texture_height,
                         0,
                         GL_RGB,
                         GL_UNSIGNED_BYTE,
                         pixelbuffer));
    }
  else
    {
      CHECKGL (glTexImage2D (GL_TEXTURE_2D,
                         0,
                         GL_RGBA8,
                         (gint) texture_width,
                         (gint) texture_height,
                         0,
                         GL_RGBA,
                         GL_UNSIGNED_BYTE,
                         pixelbuffer));
    }

  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP));

  return opengl_id;
}
/**
 * ctk_create_opengl_texture_from_file
 *
 * filename: filename of the texture to read
 * texture_width: width of the texture
 * texture_height: height of the texture
 **/
guint ctk_create_opengl_texture_from_file (const char* filename)
{
  g_return_val_if_fail (filename, 0);

  GError *error = 0;
  GdkPixbuf* pixbuf = gdk_pixbuf_new_from_file (filename, &error);
  if (error != 0)
    {
      g_warning("[ctk_create_opengl_texture_from_file] %s() at line %s: Invalid pixbuf", G_STRFUNC, G_STRLOC);
      return 0;
    }
  return ctk_create_opengl_texture_from_pixbuf (pixbuf);
}

/**
 * ctk_delete_opengl_texture
 *
 * @opengl_texture_id: OpenGL texture id to delete
 **/
void ctk_delete_opengl_texture (guint opengl_texture_id)
{
  if (opengl_texture_id != 0)
    CHECKGL (glDeleteTextures (1, &opengl_texture_id));
}

