# -*- coding: ascii -*-

###########################################################################
# clive, video extraction utility
# Copyright (C) 2007-2008 Toni Gundogdu
#
# clive is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 0.1.2-1307 USA
###########################################################################

## Tkinter based classes for GUI configuration.

import sys
import os
import subprocess
import Tkinter
import tkSimpleDialog
import tkFileDialog
import tkMessageBox

from clive.opts import Options

__all__ = ['Configure']


## The main GUI class for configuration.
class Configure:

	## Constructor
	def __init__(self):
		self._widgets = {}
	
	## Creates the widgets and enters the main loop.
	# 
	# Displays the Tkinter based GUI.
	def main(self):
		self._optparser = Options()
		(self._opts, self.args) = self._optparser.parse()
		self._root = Tkinter.Tk()
		self._root.title('clive - configure')
		self._root.geometry('+100+100')
		self._create_widgets()
		self._set_initial_values()
		self._root.mainloop()

	def _create_widgets(self):
		self._root.rowconfigure(0, weight=1)
		self._root.columnconfigure(0, weight=1)

		items = [
			('Verbose', 'verbose', None),
			('Log', 'log', None),
			('Read input from clipboard', 'from_clipb',
				self._on_clipb_changed),
		]

		self._group(
			self._root,
			group = ('General', 0, 0, 'nw', 1),
			rows = [
				(None, 0, 0, items, None, 'check'),
			]
		)

		items = [
			('Skip', '', None),
			('Overwrite', 'overwrite', None),
			('Rename', 'rename', None),
		]

		self._widgets['exists'] = Tkinter.StringVar()
		self._group(
			self._root,
			group = ('If output file exists:', 0, 1, 'ne', 1),
			rows = [
				(None, 0, 0, items, self._widgets['exists'], 'radio'),
			]
		)

		items_play = [
			('Disabled', 'off', self._on_play_changed),
			('Source', 'src', self._on_play_changed),
			('MPEG', 'mpg', self._on_play_changed),
			('AVI', 'avi', self._on_play_changed),
		]

		items_encode = [
			('Disabled', 'off', self._on_encode_changed),
			('MPEG', 'mpg', self._on_encode_changed),
			('AVI', 'avi', self._on_encode_changed),
		]

		self._widgets['play'] = Tkinter.StringVar()
		self._widgets['re_encode'] = Tkinter.StringVar()

		self._group(
			self._root,
			group = ('Action after successful extraction:',
				1, 0, 'nwe', 2),
			rows = [
				('Play extracted video:', 0, 0, items_play,
					self._widgets['play'], 'radio'),
				('Re-encode extracted video:', 1, 0, items_encode,
					self._widgets['re_encode'], 'radio'),
			]
		)

		btns = Tkinter.Frame(self._root)
		btns.grid(row=2, column=1, padx=5, pady=5, sticky='e')

		btn_items = [
			('Advanced...', self._on_more),
			('Cancel', self._root.destroy),
			('OK', self._on_exit),
		]

		for (index, btn) in enumerate(btn_items):
			(text, cmd) = btn
			b = Tkinter.Button(btns, text=text, command=cmd, width=10)
			b.grid(row=0, column=index, padx=5, pady=5, sticky='w')

	def _set_initial_values(self):
		action = '' # 'skip'
		if self._opts.overwrite: action = 'overwrite'
		elif self._opts.rename: action = 'rename'
		self._widgets['exists'].set(action)

		self._widgets['verbose'].set(self._opts.verbose)
		self._widgets['log'].set(self._opts.log)
		self._widgets['from_clipb'].set(self._opts.from_clipb)

		self._widgets['play'].set(self._opts.play)
		self._widgets['re_encode'].set(self._opts.re_encode)

	def _group(self, parent, group, rows):
		(g_text, g_row, g_col, g_sticky, g_colspan) = group

		g = Tkinter.LabelFrame(parent, text=g_text)
		g.grid(row=g_row, column=g_col, padx=10, pady=10,
			sticky=g_sticky, columnspan=g_colspan)

		g.grid_rowconfigure(0, weight=1)
		g.grid_columnconfigure(0, weight=1)

		for row in rows:
			(r_text, r_row, r_col, r_items, r_var, r_type) = row

			if r_text:
				l = Tkinter.Label(g, text=r_text)
				l.grid(row=r_row, column=r_col, sticky='w', padx=5, pady=5)

			for (index, item) in enumerate(r_items):
				(text, name, cmd) = item
				if r_type == 'radio':
					w = Tkinter.Radiobutton(g, text=text,
						variable=r_var, value=name, command=cmd)
				elif r_type == 'check':
					if not self._widgets.has_key(name):
						self._widgets[name] = Tkinter.IntVar()
					w = Tkinter.Checkbutton(g, text=text,
						variable=self._widgets[name], command=cmd)
				else:
					continue
				w.grid(row=r_row, column=r_col+index+1,
					padx=5, pady=5, sticky='w')

	def _on_clipb_changed(self):
		if sys.platform != 'win32' and self._widgets['from_clipb'].get():
			if not self._opts.xclip or len(self._opts.xclip) == 0:
				tkMessageBox.showwarning('Warning',
					'Path to `xclip` needs to be specified. ' +
					'Click "Advanced" to do that.')

	def _on_play_changed(self):
		v = self._widgets['play'].get()
		if v != 'off':
			if not self._opts.player or len(self._opts.player) == 0:
				tkMessageBox.showwarning('Warning',
					'Path to a player command needs to be specified. ' +
					'Click "Advanced" to do that.')
			if v != 'src':
				self._widgets['re_encode'].set(v)

	def _on_encode_changed(self):
		p = self._widgets['play'].get()
		r = self._widgets['re_encode'].get()
		if r != 'off':
			if not self._opts.ffmpeg or len(self._opts.ffmpeg) == 0:
				tkMessageBox.showwarning('Warning',
					'Path to `ffmpeg` needs to be specified. ' +
					'Click "Advanced" to do that.')
			if p != 'off' and p != 'src':
				self._widgets['play'].set(r)

	def _on_more(self):
		AdvancedDialog(self._root, 'Advanced settings', self._opts)

	def _on_exit(self):
		v = self._widgets['exists'].get()
		if v == 'overwrite': self._opts.overwrite = 1
		elif v == 'rename': self._opts.rename = 1
		else: self._opts.overwrite = 0; self._opts.rename = 0;

		self._opts.verbose = self._widgets['verbose'].get()
		self._opts.log = self._widgets['log'].get()
		self._opts.from_clipb = self._widgets['from_clipb'].get()

		self._opts.play = self._widgets['play'].get()
		self._opts.re_encode = self._widgets['re_encode'].get()

		self._optparser.write_config(self._opts)
		self._root.destroy()


## The class that wraps the "Advanced" options in a dialog
class AdvancedDialog(tkSimpleDialog.Dialog):

	## Constructor
	#
	# Constructs and displays the dialog. See also tkSimpleDialog.Dialog
	# documentation for more info.
	#
	# \param self The object pointer
	# \param parent The parent object
	# \param title The title for the dialog
	# \param opts The program options
	def __init__(self, parent, title, opts):
		self._opts = opts
		tkSimpleDialog.Dialog.__init__(self, parent, title)

	## Creates the dialog widgets and sets their initial values.
	def body(self, master):
		self._root = master
		self._widgets = {}
		#self._result = None

		items = [
			('proxy', 'Proxy:', 'entry', None, 1, None),
			('user_agent', 'User-agent:', 'entry', None, 1, None),
			('throttle', 'Throttle limit (KB/s):', 'entry', None, 1, None),
			('gzip', 'Use HTTP stream compression', 'check', 'w', 2, None),
		]

		self._group(master,
			group = ('Connection', 0, 0, 'nw', 1),
			items = items
		)

		items = [
			('disable_mask', 'Disable mask', 'check', None, 1,
				self._on_disable_mask),
			('output_mask', 'Mask:', 'entry', None, 1, None),
			(None, 'blank=default, "<userdef>"=~/.clive/userdef.py',
				'label', 'w', 2, None),
			('output_fmt', 'Format:', 'entry', None, 1, None),
			(None, 'blank=default; %t=title, %h=host, %i=id, %e=file ext.',
				'label', 'w', 2, None),
		]

		self._group(master,
			group = ('Output', 0, 1, 'ne', 1),
			items = items
		)

		items = [
			('prefix', 'Prefix:', 'entry', None, 1, self._on_browse_prefix),
			('player', 'Player:', 'entry', None, 1, self._on_browse_player),
			('ffmpeg', 'ffmpeg:', 'entry', None, 1, self._on_browse_ffmpeg),
		]

		if sys.platform != 'win32':
			item = ('xclip', 'xclip:', 'entry', None, 1,
				self._on_browse_xclip)
			items.append(item)
			item = (None,'Autodetect', 'button', None, 1,
				self._on_locate)
			items.append(item)

		self._group(master,
			group = ('Paths', 1, 0, 'we', 2),
			items = items
		)

		self._set_initial_values()

	## Called on either Apply or OK, applies the current widget settings.
	def apply(self):
		self._opts.proxy = self._widgets['proxy'][0].get()

		v = self._widgets['user_agent'][0].get()
		if self._opts._random_user_agent and len(v) > 0:
			self._opts._random_user_agent = 0 # Reset flag
		self._opts.user_agent = v

		v = self._widgets['throttle'][0].get()

		if len(v) == 0: v = 0
		else: v = int(v)
		self._opts.throttle = v*1024
		self._opts.gzip = self._widgets['gzip'].get()

		self._opts.output_mask = self._widgets['output_mask'][0].get()
		if self._widgets['disable_mask'].get():
			self._opts.output_mask = 'off'
		self._opts.output_fmt = self._widgets['output_fmt'][0].get()

		self._opts.prefix = self._widgets['prefix'][0].get()
		self._opts.player = self._widgets['player'][0].get()
		self._opts.ffmpeg = self._widgets['ffmpeg'][0].get()
		if sys.platform != 'win32':
			self._opts.xclip = self._widgets['xclip'][0].get()

	def _set_initial_values(self):
		if self._opts.proxy:
			self._widgets['proxy'][0].set(self._opts.proxy)
		if not self._opts._random_user_agent:
			self._widgets['user_agent'][0].set(self._opts.user_agent)
		if self._opts.throttle > 0:
			self._widgets['throttle'][0].set(str(self._opts.throttle/1024))
		self._widgets['gzip'].set(self._opts.gzip)

		if self._opts.output_mask != 'A-Za-z0-9':
			if self._opts.output_mask == 'off':
				self._widgets['disable_mask'].set(1)
				self._on_disable_mask()
			else:
				self._widgets['output_mask'][0].set(self._opts.output_mask)
		if self._opts.output_fmt != '%t.%e':
			self._widgets['output_fmt'][0].set(self._opts.output_fmt)

		if self._opts.prefix != os.getcwd():
			self._widgets['prefix'][0].set(self._opts.prefix)
		if self._opts.player:
			self._widgets['player'][0].set(self._opts.player)
		if self._opts.ffmpeg:
			self._widgets['ffmpeg'][0].set(self._opts.ffmpeg)
		if sys.platform != 'win32':
			if self._opts.xclip:
				self._widgets['xclip'][0].set(self._opts.xclip)
	"""
	def validate(self):
		return 1
		#if len(self.e.get()) == 0:
		#	return 0
		#return 1
	"""
	
	def _entry(self, parent, label, row, col=0,
			btn_cmd=None, btn_text='...'):

		l = Tkinter.Label(parent, text=label)
		l.grid(row=row, column=col, padx=5, pady=5, sticky='w')

		v = Tkinter.StringVar()
		e = Tkinter.Entry(parent, textvariable=v)
		e.grid(row=row, column=col+1, padx=5, pady=5, sticky='we')

		parent.rowconfigure(col+1, weight=1)
		parent.columnconfigure(col+1, weight=1)

		if btn_cmd:
			b = Tkinter.Button(parent, text=btn_text, command=btn_cmd)
			b.grid(row=row, column=col+2, padx=5, pady=5, sticky='e')

		return (v,e)

	def _group(self, parent, group, items):
		(g_text, g_row, g_col, g_sticky, g_colspan) = group

		g = Tkinter.LabelFrame(parent, text=g_text)
		g.grid(row=g_row, column=g_col, padx=10, pady=10,
			sticky=g_sticky, columnspan=g_colspan)

		for (index, item) in enumerate(items):
			(i_name, i_text, i_type, i_sticky, i_colspan, i_cmd) = item
			if i_type == 'entry':
				self._widgets[i_name] = \
					self._entry(g,i_text, row=index, btn_cmd=i_cmd)
			elif i_type == 'check':
				v = Tkinter.IntVar()
				w = Tkinter.Checkbutton(g, text=i_text,
					variable=v, command=i_cmd)
				w.grid(row=index, column=0, padx=5, pady=5,
					columnspan=i_colspan, sticky=i_sticky)
				self._widgets[i_name] = v
			elif i_type == 'button':
				w = Tkinter.Button(g, text=i_text, command=i_cmd)
				w.grid(row=index, column=0, padx=5, pady=5,
					columnspan=i_colspan)
			elif i_type == 'label':
				Tkinter.Label(g, text=i_text).grid(row=index, column=0,
					padx=5, pady=5, columnspan=i_colspan, sticky=i_sticky)

	def _on_browse_prefix(self):
		path = tkFileDialog.askdirectory(parent=self._root)
		if path:
			self._widgets['prefix'][0].set(path)

	def _on_browse_player(self):
		path = tkFileDialog.askopenfilename(parent=self._root)
		if path:
			self._widgets['player'][0].set(path + ' %i')

	def _on_browse_ffmpeg(self):
		path = tkFileDialog.askopenfilename(parent=self._root)
		if path:
			self._widgets['ffmpeg'][0].set(path + ' -y -i %i %o')

	def _on_browse_xclip(self):
		path = tkFileDialog.askopenfilename(parent=self._root)
		if path:
			self._widgets['xclip'][0].set(path + ' -o')

	def _on_locate(self):
		cmds = [
			('vlc', '%i', 'player'),
			('ffmpeg', '-y -i %i %o', 'ffmpeg'),
			('xclip', '-o', 'xclip'),
		]

		for cmd in cmds:
			pipe = subprocess.Popen('which %s 2>/dev/null' % cmd[0],
				shell=1, stdout=subprocess.PIPE).stdout
			data = pipe.readlines()
			if len(data) > 0:
				ln = data[0].rstrip('\n')
				if len(ln) > 0:
					self._widgets[cmd[2]][0].set(ln + ' %s' % cmd[1])
	
	def _on_disable_mask(self):
		if self._widgets['disable_mask'].get(): state = 'disabled'
		else: state = 'normal'
		self._widgets['output_mask'][1].config(state=state)


if __name__ == '__main__':
	Configure().main()


