// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef UI_VIEWS_PAINTER_H_
#define UI_VIEWS_PAINTER_H_

#include "base/basictypes.h"
#include "base/compiler_specific.h"
#include "third_party/skia/include/core/SkColor.h"
#include "ui/views/views_export.h"

namespace gfx {
class Canvas;
class ImageSkia;
class Insets;
class Rect;
class Size;
}

// A macro to define arrays of IDR constants used with CreateImageGridPainter.
#define IMAGE_GRID(x) { x ## _TOP_LEFT,    x ## _TOP,    x ## _TOP_RIGHT, \
                        x ## _LEFT,        x ## _CENTER, x ## _RIGHT, \
                        x ## _BOTTOM_LEFT, x ## _BOTTOM, x ## _BOTTOM_RIGHT, }

namespace views {

// Painter, as the name implies, is responsible for painting in a particular
// region. Think of Painter as a Border or Background that can be painted
// in any region of a View.
class VIEWS_EXPORT Painter {
 public:
  // A convenience method for painting a Painter in a particular region.
  // This translates the canvas to x/y and paints the painter.
  static void PaintPainterAt(gfx::Canvas* canvas,
                             Painter* painter,
                             const gfx::Rect& rect);

  // Creates a painter that draws a gradient between the two colors.
  static Painter* CreateHorizontalGradient(SkColor c1, SkColor c2);
  static Painter* CreateVerticalGradient(SkColor c1, SkColor c2);

  // Creates a painter that draws a multi-color gradient. |colors| contains the
  // gradient colors and |pos| the relative positions of the colors. The first
  // element in |pos| must be 0.0 and the last element 1.0. |count| contains
  // the number of elements in |colors| and |pos|.
  static Painter* CreateVerticalMultiColorGradient(SkColor* colors,
                                                   SkScalar* pos,
                                                   size_t count);

  // Creates a painter that divides |image| into nine regions. The four corners
  // are rendered at the size specified in insets (eg. the upper-left corner is
  // rendered at 0 x 0 with a size of insets.left() x insets.top()). The four
  // edges are tiled and the center is stretched to fill the destination size.
  static Painter* CreateImagePainter(const gfx::ImageSkia& image,
                                     const gfx::Insets& insets);

  // Creates a painter that paints nine images as a scalable grid. The four
  // corners are rendered in their full sizes (they are assumed to share widths
  // by column and heights by row). The four edges are tiled and the center is
  // stretched to fill the destination size.
  // |image_ids| must contain nine image IDs specified in this order: Top-Left,
  // Top, Top-Right, Left, Center, Right, Bottom-Left, Bottom, Bottom-Right.
  static Painter* CreateImageGridPainter(const int image_ids[]);

  virtual ~Painter() {}

  // Paints the painter in the specified region.
  virtual void Paint(gfx::Canvas* canvas, const gfx::Size& size) = 0;
};

// HorizontalPainter paints 3 images into a box: left, center and right. The
// left and right images are drawn to size at the left/right edges of the
// region. The center is tiled in the remaining space. All images must have the
// same height.
class VIEWS_EXPORT HorizontalPainter : public Painter {
 public:
  // Constructs a new HorizontalPainter loading the specified image names.
  // The images must be in the order left, right and center.
  explicit HorizontalPainter(const int image_resource_names[]);

  virtual ~HorizontalPainter() {}

  // Paints the images.
  virtual void Paint(gfx::Canvas* canvas, const gfx::Size& size) OVERRIDE;

  // Height of the images.
  int height() const { return height_; }

 private:
  // The image chunks.
  enum BorderElements {
    LEFT,
    CENTER,
    RIGHT
  };

  // The height.
  int height_;
  // NOTE: the images are owned by ResourceBundle. Don't free them.
  const gfx::ImageSkia* images_[3];

  DISALLOW_COPY_AND_ASSIGN(HorizontalPainter);
};

}  // namespace views

#endif  // UI_VIEWS_PAINTER_H_
