// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef SANDBOX_LINUX_SECCOMP_BPF_VERIFIER_H__
#define SANDBOX_LINUX_SECCOMP_BPF_VERIFIER_H__

#include <linux/filter.h>

#include <utility>
#include <vector>

#include "sandbox/linux/seccomp-bpf/sandbox_bpf.h"


namespace playground2 {

class Verifier {
 public:
  // Evaluate the BPF program for all possible inputs and verify that it
  // computes the correct result. We use the "evaluators" to determine
  // the full set of possible inputs that we have to iterate over.
  // Returns success, if the BPF filter accurately reflects the rules
  // set by the "evaluators".
  // Upon success, "err" is set to NULL. Upon failure, it contains a static
  // error message that does not need to be free()'d.
  static bool VerifyBPF(const std::vector<struct sock_filter>& program,
                        const Sandbox::Evaluators& evaluators,
                        const char **err);

  // Evaluate a given BPF program for a particular set of system call
  // parameters. If evaluation failed for any reason, "err" will be set to
  // a non-NULL error string. Otherwise, the BPF program's result will be
  // returned by the function and "err" is NULL.
  // We do not actually implement the full BPF state machine, but only the
  // parts that can actually be generated by our BPF compiler. If this code
  // is used for purposes other than verifying the output of the sandbox's
  // BPF compiler, we might have to extend this BPF interpreter.
  static uint32_t EvaluateBPF(const std::vector<struct sock_filter>& program,
                              const struct arch_seccomp_data& data,
                              const char **err);

 private:
  struct State {
    State(const std::vector<struct sock_filter>& p,
          const struct arch_seccomp_data& d) :
      program(p),
      data(d),
      ip(0),
      accumulator(0),
      acc_is_valid(false) {
    }
    const std::vector<struct sock_filter>& program;
    const struct arch_seccomp_data&        data;
    unsigned int                           ip;
    uint32_t                               accumulator;
    bool                                   acc_is_valid;

   private:
    DISALLOW_IMPLICIT_CONSTRUCTORS(State);
  };

  static bool     VerifyErrorCode(const std::vector<struct sock_filter>& prg,
                                  struct arch_seccomp_data *data,
                                  const ErrorCode& code, const char **err);
  static void     Ld (State *state, const struct sock_filter& insn,
                      const char **err);
  static void     Jmp(State *state, const struct sock_filter& insn,
                      const char **err);
  static uint32_t Ret(State *state, const struct sock_filter& insn,
                      const char **err);

  DISALLOW_IMPLICIT_CONSTRUCTORS(Verifier);
};

}  // namespace

#endif  // SANDBOX_LINUX_SECCOMP_BPF_VERIFIER_H__
