// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

/**
 * @fileoverview
 * Class handling creation and teardown of a remoting host session.
 *
 * This abstracts a <embed> element and controls the plugin which does the
 * actual remoting work.  There should be no UI code inside this class.  It
 * should be purely thought of as a controller of sorts.
 */

'use strict';

/** @suppress {duplicate} */
var remoting = remoting || {};

/**
 * @constructor
 */
remoting.HostSession = function() {
};

/** @type {remoting.HostPlugin} */
remoting.HostSession.prototype.plugin = null;

// Note that these values are copied directly from host_script_object.h and
// must be kept in sync.
/** @enum {number} */
remoting.HostSession.State = {
  UNKNOWN: -1,
  DISCONNECTED: 0,
  STARTING: 1,
  REQUESTED_ACCESS_CODE: 2,
  RECEIVED_ACCESS_CODE: 3,
  CONNECTED: 4,
  DISCONNECTING: 5,
  ERROR: 6,
  INVALID_DOMAIN_ERROR: 7
};

/**
 * Create an instance of the host plugin.
 * @return {remoting.HostPlugin} The new plugin instance.
 */
remoting.HostSession.createPlugin = function() {
  var plugin = document.createElement('embed');
  plugin.type = remoting.PLUGIN_MIMETYPE;
  // Hiding the plugin means it doesn't load, so make it size zero instead.
  plugin.width = 0;
  plugin.height = 0;
  return /** @type {remoting.HostPlugin} */ (plugin);
};

/**
 * Create the host plugin and initiate a connection.
 * @param {Element} container The parent element to which to add the plugin.
 * @param {string} email The user's email address.
 * @param {string} accessToken A valid OAuth2 access token.
 * @param {function(boolean):void} onNatTraversalPolicyChanged Callback
 *     for notification of changes to the NAT traversal policy.
 * @param {function(remoting.HostSession.State):void} onStateChanged
 *     Callback for notifications of changes to the host plugin's state.
 * @param {function(string):void} logDebugInfo Callback allowing the plugin
 *     to log messages to the debug log.
 */
remoting.HostSession.prototype.createPluginAndConnect =
    function(container, email, accessToken,
             onNatTraversalPolicyChanged, onStateChanged, logDebugInfo) {
  this.plugin = remoting.HostSession.createPlugin();
  container.appendChild(this.plugin);
  this.plugin.onNatTraversalPolicyChanged = onNatTraversalPolicyChanged;
  this.plugin.onStateChanged = onStateChanged;
  this.plugin.logDebugInfo = logDebugInfo;
  this.plugin.localize(chrome.i18n.getMessage);
  this.plugin.connect(email, 'oauth2:' + accessToken);
};

/**
 * Get the access code generated by the host plugin. Valid only after the
 * plugin state is RECEIVED_ACCESS_CODE.
 * @return {string} The access code.
 */
remoting.HostSession.prototype.getAccessCode = function() {
  return this.plugin.accessCode;
};

/**
 * Get the lifetime for the access code. Valid only after the plugin state is
 * RECEIVED_ACCESS_CODE.
 * @return {number} The access code lifetime, in seconds.
 */
remoting.HostSession.prototype.getAccessCodeLifetime = function() {
  return this.plugin.accessCodeLifetime;
};

/**
 * Get the email address of the connected client. Valid only after the plugin
 * state is CONNECTED.
 * @return {string} The client's email address.
 */
remoting.HostSession.prototype.getClient = function() {
  return this.plugin.client;
};

/**
 * Disconnect the client.
 * @return {void} Nothing.
 */
remoting.HostSession.prototype.disconnect = function() {
  this.plugin.disconnect();
};


/**
 * Remove the plugin element from the document.
 * @return {void} Nothing.
 */
remoting.HostSession.prototype.removePlugin = function() {
  this.plugin.parentNode.removeChild(this.plugin);
};
