// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "base/command_line.h"
#include "base/file_util.h"
#include "base/path_service.h"
#include "base/utf_string_conversions.h"
#include "content/public/browser/web_contents.h"
#include "content/public/common/content_paths.h"
#include "content/public/common/content_switches.h"
#include "content/public/test/browser_test_utils.h"
#include "content/shell/shell.h"
#include "content/test/content_browser_test.h"
#include "content/test/content_browser_test_utils.h"
#include "content/test/gpu/gpu_test_config.h"
#include "content/test/gpu/gpu_test_expectations_parser.h"
#include "net/base/net_util.h"

namespace content {

class WebGLConformanceTest : public ContentBrowserTest {
 public:
  WebGLConformanceTest() {}

  virtual void SetUpCommandLine(CommandLine* command_line) {
    // Allow privileged WebGL extensions.
    command_line->AppendSwitch(switches::kEnablePrivilegedWebGLExtensions);
  }

  virtual void SetUpInProcessBrowserTestFixture() {
    FilePath webgl_conformance_path;
    PathService::Get(base::DIR_SOURCE_ROOT, &webgl_conformance_path);
    webgl_conformance_path = webgl_conformance_path.Append(
        FILE_PATH_LITERAL("third_party"));
    webgl_conformance_path = webgl_conformance_path.Append(
        FILE_PATH_LITERAL("webgl_conformance"));
    ASSERT_TRUE(file_util::DirectoryExists(webgl_conformance_path))
        << "Missing conformance tests: " << webgl_conformance_path.value();

    PathService::Get(DIR_TEST_DATA, &test_path_);
    test_path_ = test_path_.Append(FILE_PATH_LITERAL("gpu"));
    test_path_ = test_path_.Append(FILE_PATH_LITERAL("webgl_conformance.html"));

    ASSERT_TRUE(bot_config_.LoadCurrentConfig(NULL))
        << "Fail to load bot configuration";
    ASSERT_TRUE(bot_config_.IsValid())
        << "Invalid bot configuration";

    ASSERT_TRUE(test_expectations_.LoadTestExpectations(
        GPUTestExpectationsParser::kWebGLConformanceTest));
  }

  void RunTest(const std::string& url) {
    std::string test_name =
        testing::UnitTest::GetInstance()->current_test_info()->name();
    if (StartsWithASCII(test_name, "MANUAL_", true))
      test_name = test_name.substr(strlen("MANUAL_"));
    int32 expectation =
        test_expectations_.GetTestExpectation(test_name, bot_config_);
    if (expectation != GPUTestExpectationsParser::kGpuTestPass) {
      LOG(WARNING) << "Test " << test_name << " is bypassed";
      return;
    }

    DOMMessageQueue message_queue;
    NavigateToURL(shell(), net::FilePathToFileURL(test_path_));

    std::string message;
    NavigateToURL(shell(), GURL("javascript:start('" + url + "');"));
    ASSERT_TRUE(message_queue.WaitForMessage(&message));

    EXPECT_STREQ("\"SUCCESS\"", message.c_str()) << message;
  }

 private:
  FilePath test_path_;
  GPUTestBotConfig bot_config_;
  GPUTestExpectationsParser test_expectations_;
};

#define CONFORMANCE_TEST(name, url) \
IN_PROC_BROWSER_TEST_F(WebGLConformanceTest, MANUAL_##name) { \
  RunTest(url); \
}

// The test declarations are located in webgl_conformance_test_list_autogen.h,
// because the list is automatically generated by a script.
// See: generate_webgl_conformance_test_list.py
#include "webgl_conformance_test_list_autogen.h"

}  // namespace content
