// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/search_engines/edit_search_engine_controller.h"

#include "base/string_util.h"
#include "base/utf_string_conversions.h"
#include "chrome/browser/net/url_fixer_upper.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/search_engines/template_url.h"
#include "chrome/browser/search_engines/template_url_service.h"
#include "chrome/browser/search_engines/template_url_service_factory.h"
#include "content/public/browser/user_metrics.h"
#include "googleurl/src/gurl.h"

using content::UserMetricsAction;

EditSearchEngineController::EditSearchEngineController(
    TemplateURL* template_url,
    EditSearchEngineControllerDelegate* edit_keyword_delegate,
    Profile* profile)
    : template_url_(template_url),
      edit_keyword_delegate_(edit_keyword_delegate),
      profile_(profile) {
  DCHECK(profile_);
}

bool EditSearchEngineController::IsTitleValid(
    const string16& title_input) const {
  return !CollapseWhitespace(title_input, true).empty();
}

bool EditSearchEngineController::IsURLValid(
    const std::string& url_input) const {
  std::string url = GetFixedUpURL(url_input);
  if (url.empty())
    return false;

  // Convert |url| to a TemplateURLRef so we can check its validity even if it
  // contains replacement strings.  We do this by constructing a dummy
  // TemplateURL owner because |template_url_| might be NULL and we can't call
  // TemplateURLRef::IsValid() when its owner is NULL.
  TemplateURLData data;
  data.SetURL(url);
  TemplateURL t_url(profile_, data);
  const TemplateURLRef& template_ref = t_url.url_ref();
  if (!template_ref.IsValid())
    return false;

  // If this is going to be the default search engine, it must support
  // replacement.
  if (!template_ref.SupportsReplacement() &&
      (template_url_ == TemplateURLServiceFactory::GetForProfile(profile_)->
          GetDefaultSearchProvider()))
    return false;

  // Replace any search term with a placeholder string and make sure the
  // resulting URL is valid.
  return GURL(template_ref.ReplaceSearchTerms(
      TemplateURLRef::SearchTermsArgs(ASCIIToUTF16("x")))).is_valid();
}

bool EditSearchEngineController::IsKeywordValid(
    const string16& keyword_input) const {
  string16 keyword_input_trimmed(CollapseWhitespace(keyword_input, true));
  if (keyword_input_trimmed.empty())
    return false;  // Do not allow empty keyword.
  const TemplateURL* turl_with_keyword =
      TemplateURLServiceFactory::GetForProfile(profile_)->
      GetTemplateURLForKeyword(keyword_input_trimmed);
  return (turl_with_keyword == NULL || turl_with_keyword == template_url_);
}

void EditSearchEngineController::AcceptAddOrEdit(
    const string16& title_input,
    const string16& keyword_input,
    const std::string& url_input) {
  DCHECK(!keyword_input.empty());
  std::string url_string = GetFixedUpURL(url_input);
  DCHECK(!url_string.empty());

  TemplateURLService* template_url_service =
      TemplateURLServiceFactory::GetForProfile(profile_);
  TemplateURL* existing =
      template_url_service->GetTemplateURLForKeyword(keyword_input);
  if (existing && (!edit_keyword_delegate_ || existing != template_url_)) {
    // An entry may have been added with the same keyword string while the
    // user edited the dialog, either automatically or by the user (if we're
    // confirming a JS addition, they could have the Options dialog open at the
    // same time). If so, just ignore this add.
    // TODO(pamg): Really, we should modify the entry so this later one
    // overwrites it. But we don't expect this case to be common.
    CleanUpCancelledAdd();
    return;
  }

  if (!edit_keyword_delegate_) {
    // Confiming an entry we got from JS. We have a template_url_, but it
    // hasn't yet been added to the model.
    DCHECK(template_url_);
    // TemplateURLService takes ownership of template_url_.
    template_url_service->AddWithOverrides(template_url_, title_input,
                                           keyword_input, url_string);
    content::RecordAction(UserMetricsAction("KeywordEditor_AddKeywordJS"));
  } else {
    // Adding or modifying an entry via the Delegate.
    edit_keyword_delegate_->OnEditedKeyword(template_url_, title_input,
                                            keyword_input, url_string);
  }
}

void EditSearchEngineController::CleanUpCancelledAdd() {
  if (!edit_keyword_delegate_ && template_url_) {
    // When we have no Delegate, we know that the template_url_ hasn't yet been
    // added to the model, so we need to clean it up.
    delete template_url_;
    template_url_ = NULL;
  }
}

std::string EditSearchEngineController::GetFixedUpURL(
    const std::string& url_input) const {
  std::string url;
  TrimWhitespace(TemplateURLRef::DisplayURLToURLRef(UTF8ToUTF16(url_input)),
                 TRIM_ALL, &url);
  if (url.empty())
    return url;

  // Parse the string as a URL to determine the scheme. If we need to, add the
  // scheme. As the scheme may be expanded (as happens with {google:baseURL})
  // we need to replace the search terms before testing for the scheme.
  TemplateURLData data;
  data.SetURL(url);
  TemplateURL t_url(profile_, data);
  std::string expanded_url(t_url.url_ref().ReplaceSearchTerms(
      TemplateURLRef::SearchTermsArgs(ASCIIToUTF16("x"))));
  url_parse::Parsed parts;
  std::string scheme(URLFixerUpper::SegmentURL(expanded_url, &parts));
  if (!parts.scheme.is_valid())
    url.insert(0, scheme + "://");

  return url;
}
